import { ErrorSource } from './error/source';
import { ErrorType } from './error/type';
import { MetadataEvent } from './event/metadata';
import { PlayerEvent } from './event/player';
import { ProfileEvent } from './event/profile';
import { MediaConfig } from './mediaplayer';
import { MediaSinkMode } from './mediasink';
import { ClientMessage } from './message/client';
import {
    MessageHandler,
    Quality,
    SessionData,
    WebMediaPlayer,
    WebMediaPlayerProxy,
} from './web-mediaplayer';

type MessageType = ClientMessage | PlayerEvent | MetadataEvent | ProfileEvent | '2020';

/**
 * Proxy to the C++ WebMediaPlayer class.
 * @param {object} Module Emscripten module object
 * @param {object} port   Where to call 'postMessage'. Generally 'self'.
 * @param {number} id     unique id of this instance
 * @param {object} config configure this instance of the player
 */
export class MediaPlayerProxy implements WebMediaPlayerProxy {
    private id: number;
    private port: MessageHandler;
    private player: WebMediaPlayer;

    constructor(port: MessageHandler, id: number, playerFactory: (proxy: MediaPlayerProxy) => WebMediaPlayer) {
        this.id = id;
        this.port = port;
        this.player = playerFactory(this);
        this.postMessage(PlayerEvent.PROFILE, ProfileEvent.PLAYER_LOADED_IN_WORKER);
    }

    // Emscripten feature only used for testing in debug mode
    getPointer(): number {
        // tslint:disable-next-line:no-any
        return (this.player as any).$$.ptr;
    }

    // tslint:disable-next-line: no-any
    onClientMessage(funcName: string, args: any[]) {
        this.player[funcName].apply(this.player, args);
    }

    // Send current video stats to the client
    sendStats() {
        this.postMessage(ClientMessage.STATS, this.player.getStats());
    }

    getDecodingInfo(config: MediaConfig) {
        this.postMessage(ClientMessage.GET_DECODE_INFO, config);
    }

    // Send current video stats to the client
    onSessionData(sessionData: SessionData) {
        this.postMessage(PlayerEvent.SESSION_DATA, { sessionData });
    }

    onStateChanged(state: object) {
        // Emit each state as its own event
        this.postMessage(ClientMessage.STATE_CHANGED, state);
    }

    onRebuffering() {
        this.postMessage(PlayerEvent.REBUFFERING);
    }

    onQualityChanged(quality: Quality) {
        this.postMessage(PlayerEvent.QUALITY_CHANGED, { quality });
    }

    onSeekCompleted() {
        this.postMessage(PlayerEvent.SEEK_COMPLETED);
    }

    onDurationChanged(duration: number) {
        this.postMessage(PlayerEvent.DURATION_CHANGED, { duration });
    }

    onJSONMetadata(metadataJson: string) {
        const metadata = safeParseJSON(metadataJson);
        let type: MetadataEvent | PlayerEvent;
        let data: object;

        if ('ID3' in metadata) {
            type = MetadataEvent.ID3;
            data = metadata['ID3'];
        } else if ('caption' in metadata) {
            type = MetadataEvent.CAPTION;
            data = metadata['caption'];
        } else if ('meta_enter' in metadata) {
            type = PlayerEvent.METADATA_CUE_ENTER;
            data = metadata['meta_enter'];
        } else if ('meta_exit' in metadata) {
            type = PlayerEvent.METADATA_CUE_EXIT;
            data = metadata['meta_exit'];
        }

        // Only emit if valid metadata event
        if (type) {
            this.postMessage(type, data);
        }
    }

    onMetadata(type: string, metadata: Uint8Array | string) {
        if ((metadata as Uint8Array).buffer) {
            const { buffer: data } = new Uint8Array(metadata as Uint8Array);
            this.postMessage(PlayerEvent.METADATA, { type, data }, [data]);
        } else {
            this.postMessage(PlayerEvent.METADATA, { type, data: metadata });
        }
    }

    onError(type: ErrorType, code: number, source: ErrorSource, message: string) {
        this.postMessage(PlayerEvent.ERROR, { type, code, source, message });
    }

    onRecoverableError(type: ErrorType, code: number, source: ErrorSource, message: string) {
        this.postMessage(PlayerEvent.RECOVERABLE_ERROR, { type, code, source, message });
    }

    onAnalyticsEvent(name: string, rawProperties: string) {
        const properties = safeParseJSON(rawProperties);
        this.postMessage(PlayerEvent.TRACKING, { name, properties });
    }

    // Proxies for 'MediaSink'

    configure(trackID: number, codec: string, path: string, mode: MediaSinkMode, isProtected: boolean) {
        this.postMessage(ClientMessage.CONFIGURE, {
            trackID,
            codec,
            path,
            mode,
            isProtected,
        });
    }

    reinit() {
        this.sendMediaSinkRPC('reinit');
    }

    enqueue(trackID: number, heapView: Uint8Array) {
        const { buffer } = new Uint8Array(heapView);
        this.sendMediaSinkRPC('enqueue', { trackID, buffer }, [buffer]);
    }

    endOfStream() {
        this.sendMediaSinkRPC('endOfStream');
    }

    setTimestampOffset(trackID: number, offset: number) {
        this.sendMediaSinkRPC('setTimestampOffset', { trackID, offset });
    }

    play() {
        this.sendMediaSinkRPC('play');
        this.postMessage(PlayerEvent.PROFILE, ProfileEvent.MIN_BUFFER_READY);
    }

    pause() {
        this.sendMediaSinkRPC('pause');
    }

    reset() {
        this.postMessage(ClientMessage.RESET);
    }

    remove(start: number, end: number) {
        this.sendMediaSinkRPC('remove', { start, end });
    }

    seekTo(playhead: number) {
        this.sendMediaSinkRPC('seekTo', playhead);
    }

    setPlaybackRate(rate: number) {
        this.sendMediaSinkRPC('setPlaybackRate', rate);
        this.postMessage(PlayerEvent.PLAYBACK_RATE_CHANGED);
    }

    addCue(id: number, start: number, end: number) {
        this.postMessage(ClientMessage.ADD_CUE, { id, start, end });
    }

    onMasterPlaylistRequest() {
        this.postMessage(PlayerEvent.PROFILE, ProfileEvent.HLS_MASTER_PLAYLIST_REQUEST);
    }

    onMasterPlaylistReady() {
        this.postMessage(PlayerEvent.PROFILE, ProfileEvent.HLS_MASTER_PLAYLIST_READY);
    }

    onMediaPlaylistRequest() {
        this.postMessage(PlayerEvent.PROFILE, ProfileEvent.HLS_MEDIA_PLAYLIST_REQUEST);
    }

    onMediaPlaylistReady() {
        this.postMessage(PlayerEvent.PROFILE, ProfileEvent.HLS_MEDIA_PLAYLIST_READY);
    }

    // tslint:disable-next-line: no-any
    postMessage(type: MessageType, arg?: any, transfer?: Transferable[]) {
        sendMessageToClient(this.port, this.id, type, arg, transfer);
    }

    // tslint:disable-next-line: no-any
    private sendMediaSinkRPC(name: string, arg?: any, transfer?: Transferable[]) {
        sendMessageToClient(this.port, this.id, ClientMessage.MEDIA_SINK_RPC, { name, arg }, transfer);
    }

}

// Utility to send properly structured messages to the client
export function sendMessageToClient(port: MessageHandler, id: number, type: MessageType, arg?: any, transfer?: Transferable[]) { // tslint:disable-line: no-any
    port.postMessage({
        id,
        type,
        arg,
    }, transfer);
}

function safeParseJSON(jsonStr: string) {
    try {
        return JSON.parse(jsonStr);
    } catch (e) {
        // tslint:disable-next-line: no-console
        console.error('Failed JSON parse:', jsonStr);
        return {};
    }
}
