import { browser } from './browser';
import { KEY_SYSTEMS } from './drm/constants';
import { MetadataEvent } from './event/metadata';
import { PlayerEvent } from './event/player';
import { ProfileEvent } from './event/profile';
import { RemotePlayerEvent } from './event/remote';
import { PlayerState } from './event/state';
import { BufferRange, MediaSinkListener, PlaybackError } from './mediasink';
import { MediaSinkManager } from './mediasink-manager';
import { ClientMessage } from './message/client';
import { Player } from './player';
import { TypedEventEmitter } from './typed-event-emitter';
import { BrowserContext, Quality, SessionData } from './web-mediaplayer';
import { WorkerShim } from './workershim';

export const isWasmSupported = typeof window.WebAssembly === 'object'
        && typeof window.WebAssembly.instantiate === 'function';

export const canPlay = typeof MediaSource !== 'undefined'
    && MediaSource.isTypeSupported("video/mp4; codecs='avc1.42E01E,mp4a.40.2'");

export function createWorker(workerUrl: string, wasmUrl: string) {
    let w;
    if (browser.msie) {
        w = new WorkerShim(workerUrl, window);
    } else {
        w = new Worker(URL.createObjectURL(new Blob([`importScripts('${workerUrl}')`])));
    }
    w.postMessage({
        wasmBinaryUrl: wasmUrl,
    });
    return w;
}

// Public event type
export type EventType = PlayerEvent | PlayerState | MetadataEvent | ProfileEvent | RemotePlayerEvent;

export interface MediaPlayerConfig {
    logLevel?: string;
    settings?: string;
    isQualitySupported?: (q: Quality) => boolean;
    keySystem?: string;
}

export interface NetworkProfile {
    startTime: number;
    downloadDuration: number;
    bytes: number;
    segmentDuration: number;
    firstByteLatency: number;
}

// Handle internal messages as well
type MessageType = ClientMessage | EventType;

interface CachedState {
    looping: boolean;
    autoSwitchQuality: boolean;
    playerState: PlayerState;
    quality: Quality;
    qualities: Quality[];
    duration: number;
    startOffset: number;
    sessionData: SessionData;
}

interface CachedStats {
    averageBitrate: number;
    videoBitRate: number;
    bandwidthEstimate: number;
    broadcasterLatency: number;
    transcoderLatency: number;
    networkProfile: NetworkProfile[];
}

interface MetaCue {
    id: number;
    start: number;
    end: number;
}

interface MediaVideoConfig {
    contentType: string;
    width: number;
    height: number;
    bitrate: number;
    framerate: number;
}

interface EventObject {
    name: string;
    properties: object;
}

export interface MediaConfig {
    type: string;
    video: MediaVideoConfig;
}

// Chrome 63 and Opera have an issue (crbug.com/779962) that heavily throttle video in a
// background tab while silent. So, we need to stop playback in that circumstance.
const PAUSE_HIDDEN_SILENT_TAB = (browser.chrome && browser.major === 63) || browser.opera;

const GENERIC_ERROR = 1;

const TIME_TO_LOAD_START = 'time_to_load_start';
const TIME_TO_MASTER_PLAYLIST_READY = 'time_to_master_playlist_ready';
const TIME_TO_MASTER_PLAYLIST_REQUEST = 'time_to_master_playlist_request';
const TIME_TO_MIN_BUFFER_READY = 'time_to_min_buffer_ready';
const TIME_TO_PLAY_ATTEMPT = 'time_to_play_attempt';
const TIME_TO_PLAYBACK_START = 'time_to_playback_start';
const TIME_TO_PLAYER_CORE_INIT = 'time_to_player_core_init';
const TIME_TO_VARIANT_READY = 'time_to_variant_ready';
const TIME_TO_VARIANT_REQUEST = 'time_to_variant_request';

export class MediaPlayer implements Player, MediaSinkListener {
    static createWorker = createWorker;

    /**
     * @returns {boolean} true if VP9 video is supported by the browser, false otherwise
     */
    static isVP9Supported(): boolean {
        return isMSESupported() && MediaSource.isTypeSupported('video/mp4;codecs="vp09.00.10.08"');
    }

    private static instanceId = 0;
    private ttvProperties = {};
    private readonly worker: Worker;
    private readonly id: number;
    private emitter: TypedEventEmitter<MessageType> | null;
    private seekTime: number | null;
    private paused: boolean;
    private isLoaded: boolean;
    private autoplay: boolean;
    private readonly isQualitySupported: (Quality) => boolean;
    private readonly onvisibilitychange: () => void;
    private readonly onmessage: (MessageEvent) => void;
    private readonly mediaSinkManager: MediaSinkManager;
    private readonly state: CachedState;
    private readonly stats: CachedStats;

    /** MediaPlayer constructor. This is the main export of PlayerCore
     *  @param {string} config - player configuration
     *  @param {Worker} worker - The worker instance to use
     */
    constructor(config: MediaPlayerConfig, worker: Worker) {
        this.worker = worker;
        this.id = MediaPlayer.instanceId++;
        this.emitter = new TypedEventEmitter<MessageType>();
        this.seekTime = null;
        this.paused = true;
        this.isLoaded = false;
        this.autoplay = false;
        this.isQualitySupported = config.isQualitySupported || defaultIsQualitySupported;
        this.onvisibilitychange = () => this.onVisibilityChange();
        this.onmessage = (evt) => this.onWorkerMessage(evt);
        this.mediaSinkManager = new MediaSinkManager(this);
        this.ttvProperties[TIME_TO_PLAYER_CORE_INIT] = performance.now();

        // This represents cached state from the worker. State objects
        // like this one are sent from the worker on when the player
        // state changes. This represents cached state from the worker.
        this.state = {
            looping: false,
            autoSwitchQuality: true,
        } as CachedState;
        this.stats = {} as CachedStats;
        this.resetState();

        // Setup mediaplayer-worker connections
        this.attachHandlers();

        // Create a companion instance instance in the worker.
        this.postMessage('create', [{
            settings: loadSettings(config.settings),
            logLevel: String(config.logLevel), // must be a string
            mseSupported: isMSESupported(),
            keySystem: (typeof config.keySystem !== 'undefined' ? config.keySystem : getSupportedKeySystem(browser)),
            browserContext: browser,
            codecs: MediaPlayer.isVP9Supported() ? ['vp09'] : [],
        }]);

        // Not needed anymore, but still need to emit async for backwards compatibility
        self.setTimeout(function () {
            if (this.emitter) {
                this.emitter.emit(PlayerEvent.INITIALIZED);
            }
        }.bind(this), 0);
    }

    delete(): void {
        const { visibilityChange } = getVisibilityPropNames(document);
        this.postMessage('delete');
        this.worker.removeEventListener('message', this.onmessage);
        document.removeEventListener(visibilityChange, this.onvisibilitychange);
        this.mediaSinkManager.getCurrentSink().delete();
        this.emitter = null; // Don't fire events after deletion
    }

    getHTMLVideoElement(): HTMLVideoElement {
        return this.mediaSinkManager.videoElement();
    }

    load(path: string, mediaType = ''): void {
        this.ttvProperties[TIME_TO_LOAD_START] = performance.now();
        this.postMessage('load', [path, mediaType]);
        this.resetState();
    }

    play(): void {
        this.mediaSinkManager.captureGesture();
        this.paused = false;
        this.attemptPlay();
    }

    setAutoplay(enabled: boolean) {
        this.autoplay = enabled;
    }

    pause(): void {
        this.paused = true;
        this.postMessage('pause');
    }

    isPaused(): boolean {
        return this.paused;
    }

    seekTo(time: number): void {
        // TODO: Handle seeking to very end correctly
        const MIN_DUR_FROM_END = 0.5;
        if (this.getDuration() > MIN_DUR_FROM_END) {
            time = Math.min(time, this.getDuration() - MIN_DUR_FROM_END);
        }
        time = Math.max(0, time);

        // Store seek time so we can return this position until the seek completes
        this.seekTime = time;
        this.postMessage('seekTo', [time]);
    }

    isSeeking(): boolean {
        return this.seekTime !== null;
    }

    /**
     * @return total duration of the media being played back or Infinity for an unknown or
     * potentially indefinite length stream.
     */
    getDuration(): number {
        return this.state.duration;
    }

    getStartOffset(): number {
        return this.state.startOffset || 0;
    }

    getPosition(): number {
        return (this.seekTime === null) ? this.mediaSinkManager.getCurrentSink().getCurrentTime() : this.seekTime;
    }

    getBuffered(): BufferRange {
        return this.mediaSinkManager.getCurrentSink().buffered();
    }

    getBufferDuration(): number {
        return this.mediaSinkManager.getCurrentSink().bufferDuration();
    }

    getPlayerState(): PlayerState {
        return this.state.playerState;
    }

    getVideoWidth(): number {
        return this.mediaSinkManager.videoElement().videoWidth;
    }

    getVideoHeight(): number {
        return this.mediaSinkManager.videoElement().videoHeight;
    }

    getVideoFrameRate(): number {
        return this.mediaSinkManager.getCurrentSink().framerate();
    }

    getVideoBitRate(): number {
        return this.stats.videoBitRate;
    }

    getAverageBitrate(): number {
        return this.stats.averageBitrate;
    }

    getBandwidthEstimate(): number {
        return this.stats.bandwidthEstimate;
    }

    getVersion(): string {
        return VERSION;
    }

    isLiveLowLatency(): boolean {
        return Boolean(this.state.sessionData.FUTURE);
    }

    isLooping(): boolean {
        return this.state.looping;
    }

    setLooping(loop: boolean): void {
        this.postMessage('setLooping', [loop]);
        this.state.looping = loop;
    }

    isMuted(): boolean {
        return this.mediaSinkManager.getCurrentSink().isMuted();
    }

    setMuted(mute: boolean): void {
        this.mediaSinkManager.getCurrentSink().setMuted(mute);
        this.postMessage('setMuted', [mute]);
        this.emitter.emit(PlayerEvent.MUTED_CHANGED);
    }

    setVolume(volume: number): void {
        this.mediaSinkManager.getCurrentSink().setVolume(volume);
        this.postMessage('setVolume', [volume]);
        this.emitter.emit(PlayerEvent.VOLUME_CHANGED);
    }

    getVolume(): number {
        return this.mediaSinkManager.getCurrentSink().getVolume();
    }

    getQuality(): Quality {
        return this.state.quality;
    }

    setQuality(quality: Quality, adaptive = false): void {
        this.postMessage('setQuality', [quality, adaptive]);
        this.state.autoSwitchQuality = false;
    }

    getQualities(): Quality[] {
        return this.state.qualities;
    }

    setAuthToken(token: string): void {
        this.postMessage('setAuthToken', [token]);
    }

    getAutoSwitchQuality(): boolean {
        return this.state.autoSwitchQuality;
    }

    setAutoSwitchQuality(enable: boolean): void {
        this.state.autoSwitchQuality = enable;
        this.postMessage('setAutoSwitchQuality', [enable]);
    }

    setAutoInitialBitrate(bitrate: number): void {
        this.postMessage('setAutoInitialBitrate', [bitrate]);
    }

    setAutoMaxBitrate(bitrate: number): void {
        this.postMessage('setAutoMaxBitrate', [bitrate]);
    }

    setAutoMaxVideoSize(width: number, height: number): void {
        this.postMessage('setAutoMaxVideoSize', [width, height]);
    }

    setAutoViewportSize(width: number, height: number): void {
        this.postMessage('setAutoViewportSize', [width, height]);
    }

    setAutoViewportScale(scale: number): void {
        this.postMessage('setAutoViewportScale', [scale]);
    }

    // deprecated
    // tslint:disable-next-line: no-empty
    setAutoViewportSizeEnabled(enable: boolean): void {}

    getPlaybackRate(): number {
        return this.mediaSinkManager.getCurrentSink().getPlaybackRate();
    }

    setPlaybackRate(rate: number): void {
        this.postMessage('setPlaybackRate', [rate]);
    }

    setClientId(id: string): void {
        this.postMessage('setClientId', [id]);
    }

    setDeviceId(id: string): void {
        this.postMessage('setDeviceId', [id]);
    }

    /**
     * Sets speedup rate used by player
     */
    setLiveSpeedUpRate(rate: number): void {
        this.postMessage('setLiveSpeedUpRate', [rate]);
    }

    setPlayerType(type: string): void {
        this.postMessage('setPlayerType', [type]);
    }

    setLiveMaxLatency(latency: number): void {
        this.postMessage('setLiveMaxLatency', [latency]);
    }

    setLiveLowLatencyEnabled(enable: boolean): void {
        this.postMessage('setLiveLowLatencyEnabled', [enable]);
    }

    setMinBuffer(duration: number): void {
        this.postMessage('setMinBuffer', [duration]);
    }

    setMaxBuffer(duration: number): void {
        this.postMessage('setMaxBuffer', [duration]);
    }

    setVisible(visible: boolean): void {
        // TODO deprecated remove
    }

    setAnalyticsEndpoint(url: string): void {
        this.postMessage('setAnalyticsEndpoint', [url]);
    }

    setAnalyticsSendEvents(enable: boolean): void {
        this.postMessage('setAnalyticsSendEvents', [enable]);
    }

    setBufferReplaceEnabled(enable: boolean): void {
        // TODO deprecated remove
    }

    // tslint:disable-next-line:no-any
    addEventListener(name: EventType, fn: (payload?: any) => void): void {
        this.emitter.on(name, fn);
    }

    // tslint:disable-next-line:no-any
    removeEventListener(name: EventType, fn: (payload?: any) => void): void {
        this.emitter.removeListener(name, fn);
    }

    getDroppedFrames(): number {
        return this.mediaSinkManager.getCurrentSink().droppedFrames();
    }

    getDecodedFrames(): number {
        return this.mediaSinkManager.getCurrentSink().decodedFrames();
    }

    getDecodingInfo(config: MediaConfig) {
        navigator.mediaCapabilities.decodingInfo(config).then((result) => {
            this.postMessage('onMediaDecodingInfoResult', [result]);
        });
    }

    getDisplayWidth(): number {
        return this.mediaSinkManager.getCurrentSink().getDisplayWidth();
    }

    getDisplayHeight(): number {
        return this.mediaSinkManager.getCurrentSink().getDisplayHeight();
    }

    /**
     * @deprecated use getSessionData
     * @returns {object} playback session data
     */
    getManifestInfo(): SessionData {
        return this.getSessionData();
    }

    /**
     * @returns {object} playback session data
     */
    getSessionData(): SessionData {
        return this.state.sessionData;
    }

    /**
     * @returns {number} broadcast latency in seconds
     */
    getLiveLatency(): number {
        return this.stats.broadcasterLatency;
    }

    /**
     * @deprecated use getLiveLatency
     * @returns {number} broadcast latency in milliseconds
     */
    getBroadcasterLatency(): number {
        return this.stats.broadcasterLatency * 1000;
    }

    /**
     * @deprecated use getLiveLatency
     * @returns {number} transcoder latency in milliseconds
     */
    getTranscoderLatency(): number {
        return this.stats.transcoderLatency * 1000;
    }

    getNetworkProfile(): NetworkProfile[] {
        return this.stats.networkProfile;
    }

    /**
     * @returns {boolean} Are we playing DRM content
     */
    isProtected(): boolean {
        return this.mediaSinkManager.isProtected();
    }

    startRemotePlayback(): void {
        this.postMessage('startRemotePlayback');
    }

    endRemotePlayback(): void {
        this.postMessage('endRemotePlayback');
    }

    onSinkTimeUpdate(): void {
        const sink = this.mediaSinkManager.getCurrentSink();
        if (this.seekTime === null) {
            this.postMessage('onClientSinkUpdate', [{
                currentTime: sink.getCurrentTime(),
                decodedFrames: sink.decodedFrames(),
                droppedFrames: sink.droppedFrames(),
                framerate: sink.framerate(),
            }]);
            this.emitter.emit(PlayerEvent.TIME_UPDATE);
        }
    }

    onSinkBufferUpdate(): void {
        this.emitter.emit(PlayerEvent.BUFFER_UPDATE);
    }

    onSinkDurationChanged(duration: number): void {
        this.postMessage('onClientSinkDurationChanged', [duration]);
    }

    onSinkEnded(): void {
        this.postMessage('onClientSinkEnded');
    }

    onSinkIdle(): void {
        this.postMessage('onClientSinkIdle');
    }

    onSinkPlaying(forcePlay: boolean): void {
        this.postMessage('onClientSinkPlaying');
        if (forcePlay) {
            this.play();
        }
    }

    onSinkStop(playFailed: boolean): void {
        const { hidden } = getVisibilityPropNames(document);
        if (document[hidden]) {
            // If we're stopped automatically without usage of pause api
            // Or while hidden, internally pause.
            // We'll attempt to resume when we become visible again.
            this.postMessage('pause');
        } else if (!playFailed) {
            // Playback stopped without a failed 'attepmt to play'. The video
            // element must have been updated directly, so update paused state.
            this.pause();
        } else if (!this.isMuted()) {
            // 'autoplay' can be blocked for video with sound.
            // Retry playback while muted
            this.setMuted(true);
            this.mediaSinkManager.getCurrentSink().play();
            this.emitter.emit(PlayerEvent.AUDIO_BLOCKED);
        } else {
            // 'autoplay' completely blocked. Nothing we can do other than pause
            this.pause();
            this.emitter.emit(PlayerEvent.PLAYBACK_BLOCKED);
        }
    }

    onSinkError({ value, code, message }: PlaybackError): void {
        this.postMessage('onClientSinkError', [value, code, message]);
    }

    onSinkPlay(): void {
        this.emitter.emit(PlayerEvent.PROFILE, ProfileEvent.VIDEO_ELEMENT_PLAY);
    }

    onSinkVideoDisplaySizeChanged(width: number, height: number): void {
        this.setAutoViewportSize(width, height);
    }

    onRemoteDevice(available: boolean): void {
        this.emitter.emit(available ? RemotePlayerEvent.AVAILABLE : RemotePlayerEvent.UNAVAILABLE);
    }

    onSessionError(): void {
        const value = GENERIC_ERROR;
        const code = 0;
        const message = 'Chromecast session error';
        this.postMessage('onClientSinkError', [value, code, message]);
    }

    onLoadMediaError(): void {
        const value = GENERIC_ERROR;
        const code = 0;
        const message = 'Chromecast load media failed';
        this.postMessage('onClientSinkError', [value, code, message]);
    }

    onUserCancel(): void {
        this.endRemotePlayback();
        this.emitter.emit(RemotePlayerEvent.SESSION_ENDED);
    }

    onSessionStop(): void {
        this.endRemotePlayback();
        this.emitter.emit(RemotePlayerEvent.SESSION_ENDED);
    }

    onSessionStarted(deviceName: string): void {
        this.emitter.emit(RemotePlayerEvent.SESSION_STARTED, deviceName);
    }

    private attemptPlay(): void {
        const { hidden } = getVisibilityPropNames(document);
        // Can't start playing if we're in a background tab
        // or we've loaded but haven't received 'READY'
        if (!document[hidden] && this.isLoaded) {
            this.postMessage('play');
        }
        this.ttvProperties[TIME_TO_PLAY_ATTEMPT] = performance.now();
    }

    // tslint:disable-next-line:no-any
    private postMessage(funcName: string, args?: any[]): void {
        this.worker.postMessage({
            id: this.id,
            funcName,
            args,
        });
    }

    private resetState(): void {
        // Initial values for each new session (call to 'load')
        objectAssign(this.state, {
            playerState: PlayerState.IDLE,
            quality: { name: '', group: '', codecs: '', bitrate: 0, width: 0, height: 0, framerate: 0 },
            qualities: [],
            duration: 0,
            startOffset: 0,
            sessionData: {},
        });

        // Statistics calculated in the client
        objectAssign(this.stats, {
            averageBitrate: 0,
            videoBitRate: 0,
            bandwidthEstimate: 0,
            broadcasterLatency: 0, // seconds of end-to-end latency
            transcoderLatency: 0, // seconds of latency from gotranscoder to the player
            networkProfile: [], // moving window of network stats for segment downloads
        });

        this.emitter.emit(PlayerEvent.DURATION_CHANGED, {
            duration: 0,
        });

        this.seekTime = null;
        this.isLoaded = false;
    }

    private attachHandlers(): void {
        // Attach browser event handlers
        this.worker.addEventListener('message', this.onmessage);
        document.addEventListener('visibilitychange', this.onvisibilitychange);

        // Attach internal event handlers
        const em = this.emitter;
        const updateState = (state) => objectAssign(this.state, state);

        em.on(PlayerEvent.QUALITY_CHANGED, updateState);
        em.on(PlayerEvent.AUTO_SWITCH_QUALITY_CHANGED, updateState);
        em.on(PlayerEvent.DURATION_CHANGED, updateState);
        em.on(PlayerEvent.VOLUME_CHANGED, () => this.onVolumeChanged());
        em.on(PlayerEvent.MUTED_CHANGED, () => this.onMutedChanged());
        em.on(PlayerEvent.SEEK_COMPLETED, () => this.onSeekCompleted());
        em.on(PlayerEvent.ERROR, () => this.onError());
        em.on(PlayerEvent.SESSION_DATA, (props) => this.onSessionData(props));
        em.on(PlayerEvent.TRACKING, this.onTrackingEvent.bind(this));
        em.on(PlayerEvent.PROFILE, this.onProfileEvent.bind(this));
        em.on(ClientMessage.STATS, (stats) => objectAssign(this.stats, stats));
        em.on(ClientMessage.STATE_CHANGED, (state) => this.onStateChanged(state));
        em.on(ClientMessage.MEDIA_SINK_RPC, (message) => this.mediaSinkManager.applyRPC(message));
        em.on(ClientMessage.CONFIGURE, (track) => this.mediaSinkManager.configure(track));
        em.on(ClientMessage.RESET, () => this.mediaSinkManager.reset());
        em.on(ClientMessage.ADD_CUE, (cue) => this.addCue(cue));
        em.on(ClientMessage.GET_DECODE_INFO, (config) => this.getDecodingInfo(config));
        em.on(MetadataEvent.ID3, (id3) => this.onID3(id3));
    }

    private onProfileEvent(profile: string): number {
        switch (profile) {
        case ProfileEvent.HLS_MASTER_PLAYLIST_REQUEST:
            return this.ttvProperties[TIME_TO_MASTER_PLAYLIST_REQUEST] = performance.now();
        case ProfileEvent.HLS_MASTER_PLAYLIST_READY:
            return this.ttvProperties[TIME_TO_MASTER_PLAYLIST_READY] = performance.now();
        case ProfileEvent.HLS_MEDIA_PLAYLIST_REQUEST:
            return this.ttvProperties[TIME_TO_VARIANT_REQUEST] = performance.now();
        case ProfileEvent.HLS_MEDIA_PLAYLIST_READY:
            return this.ttvProperties[TIME_TO_VARIANT_READY] = performance.now();
        default:
            return null;
        }
    }

    private onTrackingEvent(trackEvent: EventObject): void {
        if (trackEvent.name === 'video-play') {
            this.onVideoPlay(trackEvent);
        }
    }

    private onVideoPlay(trackEvent: EventObject): void {
        const videoPlayTTV = [
            TIME_TO_LOAD_START,
            TIME_TO_MASTER_PLAYLIST_READY,
            TIME_TO_MASTER_PLAYLIST_REQUEST,
            TIME_TO_MIN_BUFFER_READY,
            TIME_TO_PLAY_ATTEMPT,
            TIME_TO_PLAYBACK_START,
            TIME_TO_PLAYER_CORE_INIT,
            TIME_TO_VARIANT_READY,
            TIME_TO_VARIANT_REQUEST,
        ];

        Object.assign(trackEvent.properties, this.ttvProperties);
        for (const entry in trackEvent.properties) {
            if (videoPlayTTV.includes(entry)) {
                const originalTimestamp = trackEvent.properties[entry];
                trackEvent.properties[entry] = parseInt((originalTimestamp - trackEvent.properties[TIME_TO_PLAYER_CORE_INIT]).toString(), 10);
            }
        }
    }

    private onVolumeChanged(): void {
        const { hidden } = getVisibilityPropNames(document);
        if (PAUSE_HIDDEN_SILENT_TAB && document[hidden] && this.getVolume() === 0) {
            this.postMessage('pause');
        }
    }

    private onMutedChanged(): void {
        const { hidden } = getVisibilityPropNames(document);
        if (PAUSE_HIDDEN_SILENT_TAB && document[hidden] && this.isMuted()) {
            this.postMessage('pause');
        }
    }

    private onSeekCompleted(): void {
        this.seekTime = null;
    }

    private onError(): void {
        this.paused = true;
    }

    private onStateChanged(state): void {
        // Perform state specific updates
        switch (state.playerState) {
        case PlayerState.READY:
            // Only expose supported qualities to consumers
            const qualities = filterQualities(state.qualities, this.isQualitySupported);
            state.qualities = qualities.supported;
            qualities.unsupported.forEach((q) => this.postMessage('removeQuality', [q]));

            // We can now send 'PLAY' since we've removed any unplayable qualities
            this.isLoaded = true;

            if (this.autoplay) {
                this.play();
            }

            // Start playback if we've already tried
            if (!this.paused) {
                this.attemptPlay();
            }
            break;
        case PlayerState.ENDED:
            this.paused = true;
            break;
        default:
            break;
        }

        // Updated cached state
        objectAssign(this.state, state);

        // Emit state update event.
        this.emitter.emit(state.playerState);
    }

    private addCue({ id, start, end }: MetaCue): void {
        this.mediaSinkManager.getCurrentSink().addCue(start, end, (enter) => {
            this.postMessage('onClientSinkCue', [id, enter]);
        });
    }

    private onID3(id3): void {
        id3.forEach((frame) => {
            if (frame.id === 'TXXX' && frame.desc === 'segmentmetadata' && frame.info.length) {
                const info = safeParseJSON(frame.info[0]);
                if (info.hasOwnProperty('stream_offset')) {
                    const offset = Number(info['stream_offset']);
                    if (!isNaN(offset)) {
                        this.state.startOffset = offset - this.getPosition();
                    }
                }
            }
        });
    }

    private onVisibilityChange(): void {
        const { hidden } = getVisibilityPropNames(document);

        if (!this.paused && !document[hidden]) {
            // Start playback if we switched to 'playing' while hidden
            // May send an extraneous PLAY message if already playing
            this.attemptPlay();
        }

        // Chrome 63 bug: crbug.com/779962
        if (
            PAUSE_HIDDEN_SILENT_TAB
            && !this.paused
            && document[hidden]
            && (this.isMuted() || this.getVolume() === 0)
        ) {
            this.postMessage('pause');
        }

        this.postMessage('setVisible', [!document[hidden]]);
    }

    private onSessionData(sessionData: SessionData): void {
        objectAssign(this.state, sessionData);
    }

    // Handle state updates and RPCs from worker
    private onWorkerMessage({ data }: MessageEvent): void {
        // Ignore messages to other instances
        if (data.id === this.id) {
            this.emitter.emit(data.type, data.arg);
        }
    }
}

// utilities

// tslint:disable-next-line: no-empty
function noop() {}

function safeParseJSON(jsonStr: string): object {
    try {
        return JSON.parse(jsonStr);
    } catch (e) {
        console.error('Failed JSON parse:', jsonStr);
        return {};
    }
}

/**
 * Object.assign not available on IE11, so use this simple version
 */
function objectAssign(target: object, source: object): object {
    for (const k in source) {
        if (source.hasOwnProperty(k)) {
            target[k] = source[k];
        }
    }
    return target;
}

/**
 * @returns {boolean} true if MSE (MediaSource Extensions) is supported by the browser, false otherwise.
 */
function isMSESupported(): boolean {
    return (typeof MediaSource !== 'undefined');
}

/**
 * @returns {string} UUID of the supported key system, or empty if no supported keysystem.
 */
function getSupportedKeySystem(browser: BrowserContext): string { // tslint:disable-line: no-shadowed-variable
    // Safari
    // We've currently disabled Safari due to playback issues.
    // TODO: Add this back once Safari works
    // if (window.WebKitMediaKeys && typeof WebKitMediaKeys.isTypeSupported === 'function') {
    //     if (WebKitMediaKeys.isTypeSupported(KEY_SYSTEMS.FAIRPLAY.keySystem)) {
    //         return KEY_SYSTEMS.FAIRPLAY.uuid
    //     }
    // }

    // We can't synchronously check playready or widevine
    if (typeof navigator.requestMediaKeySystemAccess === 'function') {
        if (browser.msie || browser.msedge) {
            return KEY_SYSTEMS.PLAYREADY.uuid;
        } else {
            return KEY_SYSTEMS.WIDEVINE.uuid;
        }
    }

    return '';
}

function defaultIsQualitySupported(quality: Quality): boolean {
    // In case of variant manifest urls quality codec strings is empty and we should support those
    return quality.codecs === '' ||
        typeof MediaSource === 'undefined' || // OR Passthrough mode
        MediaSource.isTypeSupported(`video/mp4;codecs="${quality.codecs}"`); // OR Check MediaSource support
}

function filterQualities(qualities: Quality[], test: (q: Quality) => boolean) {
    const supported = [];
    const unsupported = [];

    qualities.forEach((quality) => {
        if (test(quality)) {
            supported.push(quality);
        } else {
            unsupported.push(quality);
        }
    });

    return { supported, unsupported };
}

// This dynamic require statement will bundle all files
// in the 'settings' directory, and make them each available.
// If the setting doesn't exits, we return empty (default) settings
function loadSettings(settings: string) {
    try {
        return require(`settings/${settings}.json`);
    } catch (e) {
        return {};
    }
}

// https://developer.mozilla.org/en-US/docs/Web/API/Page_Visibility_API
function getVisibilityPropNames(doc) {
    let hidden;
    let visibilityChange;
    if (typeof doc.hidden !== 'undefined') {
        hidden = 'hidden';
        visibilityChange = 'visibilitychange';
    } else if (typeof doc.msHidden !== 'undefined') {
        hidden = 'msHidden';
        visibilityChange = 'msvisibilitychange';
    } else if (typeof doc.webkitHidden !== 'undefined') {
        hidden = 'webkitHidden';
        visibilityChange = 'webkitvisibilitychange';
    }

    return { hidden, visibilityChange };
}
