import { ChromecastMediaSink } from './chromecast-mediasink';
import { DRMManager } from './drm-manager';
import { BufferRange, MediaSample, MediaSink, MediaSinkListener, PlaybackError, PlaybackRateUpdate, TrackConfig } from './mediasink';
import { MSEMediaSink } from './msemediasink';
import { PassthroughSink } from './passthroughsink';
import Promise from './promise';
import { SafeTextTrack } from './safe-text-track';

interface MediaSinkMessage {
    name: string;
    // tslint:disable-next-line: no-any
    arg: any[];
}

export class MediaSinkManager {
    private readonly video: HTMLVideoElement;
    private readonly metaTrack: SafeTextTrack;
    private readonly listener: MediaSinkListener;
    private readonly drmManager: DRMManager;
    private codecs: Record<number, string>;
    private sink: MediaSink;

    /**
     * MediaSinkManager manages MediaSink lifetimes. We create a new MediaSink
     * for each stream. We'll eventually have specialized sink for various
     * playback methods (passthough, MSE, chromecast, etc)
     * @param {MediaSinkListener} listener - a collection of callback to repond playback changes
     */
    constructor(listener: MediaSinkListener) {
        this.video = document.createElement('video');
        this.metaTrack = new SafeTextTrack(this.video);
        this.listener = listener;
        this.drmManager = new DRMManager({
            video: this.video,
            listener,
        });
        this.codecs = Object.create(null);
        this.sink = new NullMediaSink(this.video);
        ChromecastMediaSink.lookForRemotePlaybackDevices(this.listener);
    }

    delete() {
        this.sink.delete();
        this.drmManager.reset();
    }

    /**
     * Configure a track. This create a new track if no track with
     * the id exists. The first call to 'configure' after 'reset'
     * creates a new MediaSink instance.
     * @param {number} track.trackID - unique identifier for this track
     * @param {string} track.codec - codec string for this track
     * @param {string} track.path - the source url
     * @param {MediaSinkMode} track.mode - mse, passthough, or chromecast
     */
    configure(config: TrackConfig) {
        const { mode, codec, trackID } = config;

        if (this.sink instanceof NullMediaSink) {
            if (mode === 'chromecast') {
                this.sink = new ChromecastMediaSink(this.listener);
            } else if (mode === 'passthrough') {
                this.sink = new PassthroughSink(this.listener, this.video);
            } else {
                this.sink = new MSEMediaSink(this.listener, this.video, this.metaTrack);
            }
        }

        // Used cached codec if missing from config
        if (codec) {
            this.codecs[trackID] = codec;
        } else {
            config.codec = this.codecs[trackID];
        }

        this.sink.configure(config);

        // Request AuthXML if protected
        const { path, isProtected } = config;
        if (path && isProtected) {
            this.drmManager.configure(path);
        }
    }

    applyRPC({ name, arg }: MediaSinkMessage) {
        this.sink[name].call(this.sink, arg);
    }

    getCurrentSink(): MediaSink {
        return this.sink;
    }

    /**
     * Clear all media and return to initial state
     * Resets all stats such as droppedFrames, etc.
     */
    reset() {
        // Destroy current MediaSink
        this.delete();

        // Create a new sink and inform listeners of changes
        this.sink = new NullMediaSink(this.video);
        this.listener.onSinkTimeUpdate();
        this.listener.onSinkBufferUpdate();
    }

    /**
     * Get the HTMLVideoElement we render to. This is a backdoor
     * to place the element in the DOM and access unexported attributes.
     * @return {HTMLVideoElement}
     */
    videoElement(): HTMLVideoElement {
        return this.video;
    }

    /**
     * @returns {boolean} Are we playing DRM content
     */
    isProtected(): boolean {
        return this.drmManager.isProtected();
    }

    /**
     * In mobile and cellular data environment, the first call to play/pause
     * has to happen as part of the user's intent to play. This function provides
     * a noop to do this, and should be called from within a user gesture so we
     * can programmatically play later.
     */
    captureGesture() {
        if (this.video.played.length === 0) {
            Promise.resolve(this.video.play()).catch(noop);
            this.video.pause();
        }
    }
}

// A MediaSink that does nothing. We use NullMediaSink an a sentinel
// value for an uninitialized sink, so don't inherit from this type
// However we do try to capture the setVolume/setMuted calls which might happen before sink is configured
class NullMediaSink implements MediaSink {
    private readonly video: HTMLVideoElement;
    // tslint:disable:no-empty
    constructor(video: HTMLVideoElement) {
        this.video = video;
    }
    configure(config: TrackConfig) { }
    enqueue(sample: MediaSample) { }
    endOfStream() { }
    setTimestampOffset(update: PlaybackRateUpdate) { }
    play(): void { }
    pause() { }
    reinit() { }
    remove(range: BufferRange) { }
    seekTo(playhead: number) { }
    setPlaybackRate(rate: number) {
        this.video.playbackRate = rate;
    }
    setVolume(volume: number) {
        this.video.volume = volume;
    }
    addCue(start: number, end: number, onCue: (enter: boolean) => void) { }
    buffered(): BufferRange { return { start: 0, end: 0 }; }
    decodedFrames(): number { return 0; }
    droppedFrames(): number { return 0; }
    framerate(): number { return 0; }
    delete() { }
    getVolume(): number {
        return this.video.volume;
    }
    isMuted(): boolean {
        return this.video.muted;
    }
    setMuted(muted: boolean) {
        this.video.muted = muted;
    }
    getDisplayWidth(): number { return 0; }
    getDisplayHeight(): number { return 0; }
    getPlaybackRate(): number {
        return this.video.playbackRate;
    }
    getCurrentTime(): number { return 0; }
    bufferDuration(): number { return 0; }
    // tslint:enable:no-empty
}

// tslint:disable-next-line:no-empty
function noop() { }
