import { BufferRange, MediaSample, MediaSink, MediaSinkListener, PlaybackError, PlaybackRateUpdate, TrackConfig } from './mediasink';
import { MIN_PLAYABLE_BUFFER, PlaybackMonitor } from './playback-monitor';
import Promise from './promise';
import { SafeSourceBuffer } from './safe-source-buffer';
import { SafeTextTrack } from './safe-text-track';
import { bufferedRange, decodedFrames, droppedFrames } from './video-element-helper';

// constants
const NOT_SUPPORTED = 4; // HTMLMediaElement error code

export class MSEMediaSink implements MediaSink {
    private video: HTMLVideoElement;
    private metaTrack: SafeTextTrack;
    private codecs: Record<number, string>;
    private readonly listener: MediaSinkListener;
    private readonly playbackMonitor: PlaybackMonitor;
    private mediaSource: Promise<MediaSource> | null;
    private tracks: Record<number, Promise<SafeSourceBuffer>>;
    private trackOffsets: Record<number, number>;

    /**
     * MediaSink implements the "MediaSink" interface in javascript
     * This unfortunatly has to live in the client since it must access
     * the DOM. MediaSink handles all MSE logic.
     * @param {function()} config.ontimeupdate - playhead position has changed
     * @param {function()} config.onbufferupdate - buffered range has changed
     * @param {function()} config.onended - fired when playback ended
     * @param {function()} config.onidle - fired when playback interrupted while playing
     * @param {function()} config.onstop - fired when playback paused by browers
     * @param {function(MediaError)} config.onerror - video error with error code
     */
    constructor(listener: MediaSinkListener, video: HTMLVideoElement, metaTrack: SafeTextTrack) {
        this.video = video;
        this.metaTrack = metaTrack;
        this.codecs = Object.create(null);
        this.listener = listener;
        this.playbackMonitor = new PlaybackMonitor(video, listener);
        this.mediaSource = null;
        this.tracks = Object.create(null);
        this.trackOffsets = Object.create(null);
    }

    /**
     * Configure a track. This create a new track if no track with
     * the id exists. We'll create a new MediaSource and attach it
     * on the first call after 'reset'.
     * @param {number} track.trackID - unique identifier for this track
     * @param {string} track.codec - codec string for this track
     * @param {string} track.path - the source url
     * @param {bool} track.isProtected - Is this stream protected by DRM
     * @param {MediaSinkMode} track.mode - chromecast, mse, passthough
     */
    configure({ trackID, codec }: TrackConfig) {
        this.codecs[trackID] = codec;

        // Lazily attach a new MediaSource
        if (!this.mediaSource) {
            this.mediaSource = createMediaSource(this.video);
        }

        // Add track if it doesn't exist
        if (!this.tracks[trackID]) {
            this.addSourceBuffer(trackID, codec, 0);
        }
    }

    /**
     * Enqueue an buffer
     * @param {TypedArray} sample.buffer - fmp4 buffer
     * @param {number} sample.trackID - designates the track to update.
     */
    enqueue({ trackID, buffer }: MediaSample) {
        const track = this.tracks[trackID];
        if (track) {
            // Error already sent when track failed to configure
            track.then((srcBuf) => srcBuf.appendBuffer(buffer), noop);
        }
    }

    /**
     * Mark that we've appended the final segment
     */
    endOfStream() {
        this.scheduleUpdate().then(
            () => this.playbackMonitor.endOfStream());
    }

    /**
     * Update the timstamp offset of all future samples in this buffer
     * @param {number} update.offset - offset to apply in seconds
     * @param {number} update.trackID - designates the track to update
     */
    setTimestampOffset({ trackID, offset }: PlaybackRateUpdate) {
        const track = this.tracks[trackID];
        if (track) {
            this.trackOffsets[trackID] = offset;
            // Error already sent when track failed to configure
            track.then((srcBuf) => srcBuf.setTimestampOffset(offset), noop);
        }
    }

    /**
     * Start/resume playback
     */
    play() {
        this.scheduleUpdate().then(() => this.playbackMonitor.play());
    }

    /**
     * Stop playback
     */
    pause() {
        this.scheduleUpdate().then(() => this.playbackMonitor.pause());
    }

    /**
     * Reinit creates a new MediaSource with the same number of tracks
     * This effectively 'resets' all of the tracks, but keeps codec info,
     * timestampOffset, and the current playhead. This is needed when transitioning
     * from unprotected content to protected content (Stitched preroll CVP-2500).
     */
    reinit() {
        this.playbackMonitor.pause();
        const playhead = this.video.currentTime;

        // Remove cues in current buffer
        const buffered = this.buffered();
        this.metaTrack.remove(buffered.start, buffered.end);

        const prevSrc = this.video.src;
        this.mediaSource = createMediaSource(this.video);
        URL.revokeObjectURL(prevSrc);

        // tslint:disable-next-line:forin
        for (const trackID in this.tracks) {
            this.addSourceBuffer(trackID, this.codecs[trackID], this.trackOffsets[trackID]);
        }

        this.video.currentTime = playhead;
    }

    /**
     * Remove a range of buffered media
     * @param {number} range.start - start of range to remove
     * @param {number} range.end - end of range to remove
     */
    remove({ start, end }: BufferRange) {
        const removeRange = (srcBuf) => srcBuf.remove(start, end);
        // tslint:disable-next-line:forin
        for (const trackID in this.tracks) {
            this.tracks[trackID].then(removeRange);
        }
        // Handle metadata tracks explicitly
        this.metaTrack.remove(start, end);
    }

    /**
     * Move to playhead to a new position
     * @param {number} playhead - position of new playhead
     */
    seekTo(playhead: number) {
        const { start, end } = bufferedRange(this.video, MIN_PLAYABLE_BUFFER);
        if (playhead >= start && playhead < end) {
            // If we're seeking within the buffer, wait for pending operations
            this.scheduleUpdate().then(() => this.video.currentTime = playhead);
        } else {
            this.video.currentTime = playhead;
        }
    }

    setVolume(volume: number) {
        this.video.volume = volume;
    }

    getVolume(): number {
        return this.video.volume;
    }

    isMuted(): boolean {
        return this.video.muted;
    }

    setMuted(muted: boolean) {
        this.video.muted = muted;
    }

    getDisplayWidth(): number {
        return this.video.clientWidth;
    }

    getDisplayHeight(): number {
        return this.video.clientHeight;
    }

    /**
     * Set the playback rate
     * @param {number} rate - multiplier to the default rate
     */
    setPlaybackRate(rate: number) {
        this.video.playbackRate = rate;
    }

    getPlaybackRate() {
        return this.video.playbackRate;
    }

    getCurrentTime(): number {
        return this.video.currentTime;
    }

    /**
     * Add a cue that will be fired when the playhead crosses
     * the specified time. Only fired once.
     * @param {number} start - when the cue begins
     * @param {number} end - when the cue ends
     * @param {function} onCue - called when the cue is fired
     */
    addCue(start: number, end: number, onCue: (enter: boolean) => void) {
        // endTime musst be larger than startTime on Edge
        if (end <= start) {
            end = start + MIN_PLAYABLE_BUFFER;
        }
        // `const VTTCue` was in closure scope. However, with videojs hls tech,
        // gloval.VTTCue is temporarily polyfilled on Edge and then restored.
        // So avoiding closure scope. More details - https://jira.twitch.com/browse/CVP-2513
        // tslint:disable-next-line:no-any
        const VTTCue = (window as any).VTTCue || (window as any).TextTrackCue;
        const cue = new VTTCue(start, end, '');
        cue.onenter = () => onCue(true);
        cue.onexit = () => onCue(false);
        this.metaTrack.addCue(cue);
    }

    /**
     * @return {Number} buffered.start - start of current buffer
     * @return {Number} buffered.end - end of current buffer
     */
    buffered(): BufferRange {
        return bufferedRange(this.video, MIN_PLAYABLE_BUFFER);
    }

    /**
     * @return {Number} Duration of the current buffer
     */
    bufferDuration(): number {
        const { start, end } = this.buffered();
        return end - Math.max(start, this.video.currentTime);
    }

    /**
     * Frames decoded in this stream. Reset in 'reset'
     * @return {number} num decoded frames
     */
    decodedFrames(): number {
        return decodedFrames(this.video);
    }

    /**
     * Frames dropped in this stream. Reset in 'reset'
     * @return {number} num dropped frames
     */
    droppedFrames(): number {
        return droppedFrames(this.video);
    }

    /**
     * @return {number} framerate in fps
     */
    framerate(): number {
        return this.playbackMonitor.framerate();
    }

    /**
     * Cleanup and shutdown. No functions on MediaSink may be called
     * on the instance after this.
     */
    delete() {
        this.playbackMonitor.delete();
        this.metaTrack.remove(0, Infinity);

        // Detach MediaSource
        if (this.video.src) {
            const src = this.video.src;
            this.video.removeAttribute('src');
            this.video.load(); // https://github.com/w3c/media-source/issues/53
            URL.revokeObjectURL(src);
        }
    }

    /**
     * Create a source buffer and add it to our MediaSource
     * @param {Number} trackID to identify this new buffer
     * @param {String} codec   information for this track
     * @param {Number} offset  Initial timestamp offset
     */
    private addSourceBuffer(trackID: number | string, codec: string, offset: number) {
        const track = this.mediaSource.then((mediaSource) => {
            const srcBuf = mediaSource.addSourceBuffer(`video/mp4;${codec}`);
            srcBuf.timestampOffset = offset;
            return new SafeSourceBuffer(this.video, srcBuf);
        });

        // Want to leave promise in tracks map rejected
        // to block any future calls to 'enqueue'
        track.catch((e) => this.listener.onSinkError({
            value: NOT_SUPPORTED,
            code: NOT_SUPPORTED,
            message: e.toString(),
        }));

        this.tracks[trackID] = track;
    }

    /**
     * Schedule an update in all tracks. This allows us to order operations
     * on the video element with operations on the SourceBuffers
     * @return {Promise} Resolves when the update can be applied
     */
    private scheduleUpdate() {
        function schedulePromise(track) {
            return new Promise((resolve) => track.schedule(resolve));
        }
        const promises = [];
        // tslint:disable-next-line:forin
        for (const trackID in this.tracks) {
            promises.push(this.tracks[trackID].then(schedulePromise));
        }

        // Catch here in case SourceBuffer creation fails.
        // No need to schedule in this case
        return Promise.all(promises).catch(noop);
    }
}

// tslint:disable-next-line:no-empty
function noop() { }

/**
 * Create a MediaSource and attach it to the video element
 * @param  {HTMLVideoElement} video - the video element to attach to
 * @return {Promise(MediaSource)} Resolves when the MediaSource has been succesfully opened
 */
function createMediaSource(video: HTMLVideoElement): Promise<MediaSource> {
    return new Promise((resolve, reject) => {
        const mediaSource = new MediaSource();
        const onSourceOpen = () => {
            mediaSource.removeEventListener('sourceopen', onSourceOpen);
            // This can be fired even if the source has already been removed
            if (mediaSource.readyState === 'open') {
                // tslint:disable-next-line:no-any
                mediaSource.duration = +Infinity;
                resolve(mediaSource);
            } else {
                reject(new Error('MediaSource not opened'));
            }
        };
        mediaSource.addEventListener('sourceopen', onSourceOpen);
        const { playbackRate } = video;
        video.src = URL.createObjectURL(mediaSource);
        video.playbackRate = playbackRate;
    });
}

export const _testExports = {
    createMediaSource,
};
