import { TrackConfig } from './mediasink';
import { PassthroughSink } from './passthroughsink';
import { getFakeMediaSinkListener } from './test-utils/fake-helper';

function setupPassthroughSink() {
    const videoElement = document.createElement('video');
    const listener = getFakeMediaSinkListener();
    const specimen = new PassthroughSink(listener, videoElement);

    return { videoElement, listener, specimen };
}

describe('Test public api', () => {
    test('configure', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();

        // Test
        const samplePath = 'https://somehost/somepath/somefile';
        specimen.configure({ path: samplePath } as TrackConfig);
        expect(videoElement.src).toEqual(samplePath);
    });

    describe('play', () => {
        test('Listener callbacks', () => {
            const { videoElement, listener, specimen } = setupPassthroughSink();

            // Setup
            const spyOnPlaying = jest.spyOn(listener, 'onSinkPlaying');
            const spyOnPlay = jest.spyOn(listener, 'onSinkPlay');
            videoElement.play = () => {
                return new Promise((resolve) => {
                    videoElement.dispatchEvent(new Event('play'));
                    resolve();
                    videoElement.dispatchEvent(new Event('playing'));
                });
            };
            const spyOnVideoPlay = jest.spyOn(videoElement, 'play');

            // Test
            specimen.play();
            expect(spyOnVideoPlay).toBeCalled();
            expect(spyOnPlay).toBeCalled();
            expect(spyOnPlaying).toBeCalled();
        });

        test('When Live video and buffered region behind playhead, playhead should move to that region', () => {
            const { videoElement, listener, specimen } = setupPassthroughSink();

            // Setup video element to have the playhead past the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 60;
            const CURRENT_PLAYHEAD = 100;

            jest.spyOn(videoElement, 'duration', 'get').mockReturnValue(Infinity);
            videoElement.currentTime = CURRENT_PLAYHEAD;
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });
            videoElement.play = jest.fn();

            // Test
            specimen.play();
            expect(videoElement.currentTime).toEqual(BUFFER_START_POSITION);
        });

        test('Buffer region ahead of playhead', () => {
            const { videoElement, listener, specimen } = setupPassthroughSink();

            // Setup video element to have the playhead past the buffered region
            const BUFFER_START_POSITION = 150;
            const BUFFER_END_POSITION = 160;
            const CURRENT_PLAYHEAD = 100;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });
            videoElement.play = jest.fn();

            // Test
            specimen.play();
            expect(videoElement.currentTime).toEqual(BUFFER_START_POSITION);
        });

        test('when play() call fails the promise onSinkStop should be called with playFailed=true', () => {
            const { videoElement, listener, specimen } = setupPassthroughSink();

            // Setup
            const spyOnStop = jest.spyOn(listener, 'onSinkStop');
            videoElement.play = () => {
                return Promise.reject();
            };

            // Test
            specimen.play();
            // setImmediate necessary because the listener is fired in failed promise's catch
            setTimeout(() => {
                expect(spyOnStop).toBeCalledWith(true);
            }, 0);
        });

        test('When play() call fails because of error onSinkStop should not be called', () => {
            const { videoElement, listener, specimen } = setupPassthroughSink();

            // Setup
            const spyOnStop = jest.spyOn(listener, 'onSinkStop');
            videoElement.play = () => {
                Object.defineProperty(videoElement, 'error', {
                    value: {},
                });
                return Promise.reject();
            };

            // Test
            specimen.play();
            // setImmediate necessary because the listener is fired in failed promise's catch
            setTimeout(() => {
                expect(spyOnStop).not.toBeCalled();
            }, 0);
        });

        test('Buffer region ahead of playhead and src .mp4, then do not adjust the playhead', () => {
            const { videoElement, listener, specimen } = setupPassthroughSink();

            // Setup video element to have the playhead past the buffered region
            const BUFFER_START_POSITION = 150;
            const BUFFER_END_POSITION = 160;
            const CURRENT_PLAYHEAD = 100;

            videoElement.src = 'random.mp4';
            videoElement.currentTime = CURRENT_PLAYHEAD;
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });
            videoElement.play = jest.fn();

            // Test
            specimen.play();
            expect(videoElement.currentTime).toEqual(CURRENT_PLAYHEAD);
        });
    });

    describe('pause', () => {
        test('pause() API paused the videoelement', () => {
            const { videoElement, listener, specimen } = setupPassthroughSink();

            // Setup
            videoElement.pause = jest.fn();
            const spyOnVideoPause = jest.spyOn(videoElement, 'pause');

            // Test
            specimen.pause();
            expect(spyOnVideoPause).toBeCalled();
        });

        test('if video element pauses without the pause() API expect onSinkStop with playFailed=false', () => {
            const { videoElement, listener, specimen } = setupPassthroughSink();

            // Setup
            videoElement.play = () => {
                return new Promise((resolve) => {
                    videoElement.dispatchEvent(new Event('play'));
                    resolve();
                    videoElement.dispatchEvent(new Event('playing'));
                });
            };
            videoElement.pause = () => {
                return new Promise((resolve) => {
                    resolve();
                    Object.defineProperty(videoElement, 'paused', {
                        value: true,
                    });
                    videoElement.dispatchEvent(new Event('pause'));
                });
            };
            const spyOnStop = jest.spyOn(listener, 'onSinkStop');

            // Test
            specimen.play();

            videoElement.pause(); // Internal pause
            expect(spyOnStop).toBeCalledWith(false);
        });
    });

    test('reinit', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        videoElement.load = jest.fn();
        const spyOnLoad = jest.spyOn(videoElement, 'load');
        const samplePath = 'https://somehost/somepath/somefile';
        specimen.configure({ path: samplePath } as TrackConfig);

        // Test
        specimen.reinit();
        expect(spyOnLoad).toBeCalled();
        expect(videoElement.src).toEqual(samplePath);
    });

    test('seekTo', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_SEEK_TIME = 100;
        // Test
        specimen.seekTo(SAMPLE_SEEK_TIME);
        expect(videoElement.currentTime).toEqual(SAMPLE_SEEK_TIME);
    });

    test('setPlaybackRate', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_PLAYBACK_RATE = 1.5;
        // Test
        specimen.setPlaybackRate(SAMPLE_PLAYBACK_RATE);
        expect(videoElement.playbackRate).toEqual(SAMPLE_PLAYBACK_RATE);
    });

    test('setVolume', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_VOLUME = 0.5;
        // Test
        specimen.setVolume(SAMPLE_VOLUME);
        expect(videoElement.volume).toEqual(SAMPLE_VOLUME);
    });

    test('getVolume', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_VOLUME = 0.5;
        // Test
        videoElement.volume = SAMPLE_VOLUME;
        expect(specimen.getVolume()).toEqual(SAMPLE_VOLUME);
    });

    test('delete', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        videoElement.load = jest.fn();
        const spyOnLoad = jest.spyOn(videoElement, 'load');
        const samplePath = 'https://somehost/somepath/somefile';
        specimen.configure({ path: samplePath } as TrackConfig);

        // Test
        specimen.delete();
        expect(spyOnLoad).toBeCalled();
        expect(videoElement.src.length).toEqual(0);
    });

    test('isMuted', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_MUTED = true;
        videoElement.muted = SAMPLE_MUTED;
        // Test
        expect(specimen.isMuted()).toEqual(SAMPLE_MUTED);
    });

    test('setMuted', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_MUTED = true;
        // Test
        specimen.setMuted(SAMPLE_MUTED);
        expect(videoElement.muted).toEqual(SAMPLE_MUTED);
    });

    test('getDisplayWidth', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_WIDTH = 100;
        // Test
        Object.defineProperty(videoElement, 'clientWidth', {
            value: SAMPLE_WIDTH,
        });
        expect(specimen.getDisplayWidth()).toEqual(SAMPLE_WIDTH);
    });

    test('getDisplayHeight', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_HEIGHT = 100;
        // Test
        Object.defineProperty(videoElement, 'clientHeight', {
            value: SAMPLE_HEIGHT,
        });
        expect(specimen.getDisplayHeight()).toEqual(SAMPLE_HEIGHT);
    });

    test('getPlaybackRate', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_PLAYBACK_RATE = 1.5;
        // Test
        videoElement.playbackRate = SAMPLE_PLAYBACK_RATE;
        expect(specimen.getPlaybackRate()).toEqual(SAMPLE_PLAYBACK_RATE);
    });

    test('getCurrentTime', () => {
        // Setup
        const { videoElement, specimen } = setupPassthroughSink();
        const SAMPLE_SEEK_TIME = 100;
        // Test
        videoElement.currentTime = SAMPLE_SEEK_TIME;
        expect(specimen.getCurrentTime()).toEqual(SAMPLE_SEEK_TIME);
    });

    test('bufferDuration', () => {
        const { videoElement, specimen } = setupPassthroughSink();

        const BUFFER_START_POSITION = 50;
        const CURRENT_PLAYHEAD = 100;
        const BUFFER_END_POSITION = 120;

        videoElement.currentTime = CURRENT_PLAYHEAD;
        jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
            length: 1,
            start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
            end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
        });

        expect(specimen.bufferDuration()).toEqual(BUFFER_END_POSITION - CURRENT_PLAYHEAD);
    });

    test.skip('buffered', () => {
        // Refer video-element-helper tests
    });
    test.skip('decodedFrames', () => {
        // Refer video-element-helper tests
    });

    test.skip('droppedFrames', () => {
        // Refer video-element-helper tests
    });

    test.skip('framerate', () => {
        // Difficult to maintain difference in performance.now()
    });
});
