import { BufferRange, MediaSink, MediaSinkListener, TrackConfig } from './mediasink';
import {
    bufferedRange,
    decodedFrames,
    droppedFrames,
    getNextPlayablePosition,
    MIN_PLAYABLE_BUFFER,
    subscribe,
    willBuffer,
} from './video-element-helper';

const ERR_BUFFERING_TIMEOUT = 101;
const BUFFERING_TIMEOUT = 120000;
const HEARTBEAT_INTERVAL = 2000; // Interval to check for stalled

export class PassthroughSink implements MediaSink {
    private readonly listener: MediaSinkListener;
    private readonly videoElement: HTMLVideoElement;

    private unsubList: Function[];
    private fps: number;
    private intervalId: number;
    private paused: boolean;

    constructor(listener: MediaSinkListener, videoElement: HTMLVideoElement) {
        this.listener = listener;
        this.videoElement = videoElement;
        this.unsubList = [];
        this.fps = 0;
        this.intervalId = -1;
        this.paused = true;

        this.unsubList.push(
            subscribe(this.videoElement, 'waiting', () => this.onVideoWaiting()),
            subscribe(this.videoElement, 'timeupdate', () => this.onVideoTimeUpdate()),
            subscribe(this.videoElement, 'durationchange', () => this.onVideoDurationChange()),
            subscribe(this.videoElement, 'error', () => this.onVideoError()),

            subscribe(this.videoElement, 'play', () => this.onVideoPlay()),
            subscribe(this.videoElement, 'pause', () => this.onVideoPause()),
            subscribe(this.videoElement, 'ended', () => this.onVideoEnded()),
            subscribe(this.videoElement, 'playing', () => this.onVideoPlaying()),
        );
    }

    configure({ path }: TrackConfig): void {
        this.videoElement.src = path;
    }

    play(): void {
        const buffered = this.videoElement.buffered;
        if (buffered.length > 0) {
            const start = buffered.start(buffered.length - 1);
            const end = buffered.end(buffered.length - 1);
            if (
                // Buffered content behind playhead only for live
                (
                    this.videoElement.duration === Infinity &&
                    end < this.videoElement.currentTime
                ) ||
                // OR Buffered content ahead of playhead and not mp4
                (
                    this.videoElement.src.indexOf('.mp4') === -1 &&
                    this.videoElement.currentTime < start
                )
            ) {
                console.warn('Moving to buffered region');
                this.videoElement.currentTime = start;
            }
        }

        this.paused = false;
        Promise.resolve(this.videoElement.play()).catch(() => this.checkStopped(true));
        this.listener.onSinkPlay();
    }

    pause(): void {
        this.paused = true;
        this.videoElement.pause();
        clearTimeout(this.intervalId);
    }

    reinit(): void {
        const currentSrc = this.videoElement.src;
        this.videoElement.src = '';
        this.videoElement.load();
        this.videoElement.src = currentSrc;
    }

    seekTo(playhead: number): void {
        this.videoElement.currentTime = playhead;
    }

    setPlaybackRate(rate: number): void {
        this.videoElement.playbackRate = rate;
    }

    setVolume(volume: number): void {
        this.videoElement.volume = volume;
    }

    getVolume(): number {
        return this.videoElement.volume;
    }

    buffered(): BufferRange {
        return bufferedRange(this.videoElement, MIN_PLAYABLE_BUFFER);
    }

    decodedFrames(): number {
        return decodedFrames(this.videoElement);
    }

    droppedFrames(): number {
        return droppedFrames(this.videoElement);
    }

    framerate(): number {
        return this.fps;
    }

    delete(): void {
        this.unsubList.forEach((fn) => fn());
        this.videoElement.src = '';
        this.videoElement.load();
    }

    isMuted(): boolean {
        return this.videoElement.muted;
    }

    setMuted(muted: boolean): void {
        this.videoElement.muted = muted;
    }

    getDisplayWidth(): number {
        return this.videoElement.clientWidth;
    }

    getDisplayHeight(): number {
        return this.videoElement.clientHeight;
    }

    getPlaybackRate(): number {
        return this.videoElement.playbackRate;
    }

    getCurrentTime(): number {
        return this.videoElement.currentTime;
    }

    bufferDuration(): number {
        const { start, end } = this.buffered();
        return end - Math.max(start, this.videoElement.currentTime);
    }

    enqueue(): void {
        // no-op
    }

    endOfStream(): void {
        // no-op
    }

    setTimestampOffset(): void {
        // no-op
    }

    addCue(start: number, end: number, onCue: (enter: boolean) => void): void {
        // no-op
    }

    remove(): void {
        // no-op
    }

    private onVideoWaiting() {
        const buffered = bufferedRange(this.videoElement, MIN_PLAYABLE_BUFFER);
        if (buffered.end - this.videoElement.currentTime < MIN_PLAYABLE_BUFFER) {
            this.listener.onSinkIdle();

            // Start a timeout for buffering
            const bufferingTimeout = self.setTimeout(() => {
                this.listener.onSinkError({
                    value: ERR_BUFFERING_TIMEOUT,
                    code: ERR_BUFFERING_TIMEOUT,
                    message: 'Buffering timeout',
                });
            }, BUFFERING_TIMEOUT);

            // Clear the timeout if there is a timeupdate
            const nextTimeupdate = subscribe(this.videoElement, 'timeupdate', () => {
                nextTimeupdate();
                clearTimeout(bufferingTimeout);
            });
        }

        const playingOnTimeupdate = subscribe(this.videoElement, 'timeupdate', () => {
            // BROWSER BUG: Safari isn't emitting 'playing' when resuming from 'waiting'
            if (this.videoElement.readyState === 4) {
                playingOnTimeupdate();
                this.onVideoPlaying();
            }
        });
    }

    private onVideoTimeUpdate() {
        this.listener.onSinkTimeUpdate();

        this.listener.onSinkVideoDisplaySizeChanged(
            this.videoElement.clientWidth * self.devicePixelRatio,
            this.videoElement.clientHeight * self.devicePixelRatio,
        );
    }

    private onVideoDurationChange() {
        this.listener.onSinkDurationChanged(this.videoElement.duration);
    }

    private onVideoError() {
        const { code, message = '' } = this.videoElement.error;
        // TODO Check for decode error and reload video element once.
        this.listener.onSinkError({
            value: code,
            code,
            message,
        });
    }

    private onVideoPlay() {
        const currentPlayhead = this.videoElement.currentTime;
        clearTimeout(this.intervalId);
        this.intervalId = self.setTimeout(() => this.heartbeat(currentPlayhead), HEARTBEAT_INTERVAL);
    }

    private onVideoPause() {
        clearTimeout(this.intervalId);
        this.checkStopped(false);
    }

    private onVideoEnded() {
        this.listener.onSinkEnded();
    }

    private onVideoPlaying() {
        this.listener.onSinkPlaying(this.paused);

        // Track fps
        this.trackFPS(decodedFrames(this.videoElement), performance.now());

        // Track buffer change
        this.trackBufferUpdate(bufferedRange(this.videoElement, MIN_PLAYABLE_BUFFER).end);
    }

    /**
     * A heartbeat that monitors playback progress and corrects any playback issues.
     */
    private heartbeat(lastPlayhead: number) {
        let currentPlayhead = this.videoElement.currentTime;
        if (currentPlayhead === lastPlayhead) {
            if (willBuffer(this.videoElement, MIN_PLAYABLE_BUFFER)) {
                this.listener.onSinkIdle();
                return;
            } else {
                const nextPlayablePosition = getNextPlayablePosition(this.videoElement, MIN_PLAYABLE_BUFFER);
                if (nextPlayablePosition !== currentPlayhead) {
                    console.warn(`jumping ${nextPlayablePosition - currentPlayhead}s gap`);
                    this.videoElement.currentTime = nextPlayablePosition;
                    currentPlayhead = this.videoElement.currentTime;
                }
            }
        }

        this.intervalId = self.setTimeout(() => this.heartbeat(currentPlayhead), HEARTBEAT_INTERVAL);
    }

    /**
     * Notify mediaplayer if the browser paused on its own. This can happen
     * when the player is muted and hidden.
     */
    private checkStopped(playFailed: boolean) {
        if (this.videoElement.paused && !this.videoElement.ended && !this.videoElement.error && !this.paused) { // An auto pause scenario
            this.listener.onSinkStop(playFailed);
        }
    }

    private trackFPS(lastDecodedFrames: number, lastTime: number) {
        const currentDecodedFrames = decodedFrames(this.videoElement);
        const currentTime = performance.now();
        this.fps = (currentDecodedFrames - lastDecodedFrames) / (currentTime - lastTime) * 1000;

        const unsub = subscribe(this.videoElement, 'timeupdate', () => {
            unsub();
            this.trackFPS(currentDecodedFrames, currentTime);
        });
    }

    private trackBufferUpdate(lastBufferEnd: number) {
        const { end } = this.buffered();
        if (end !== lastBufferEnd) {
            this.listener.onSinkBufferUpdate();
        }
        const unsub = subscribe(this.videoElement, 'timeupdate', () => {
            unsub();
            this.trackBufferUpdate(end);
        });
    }
}
