import { BufferRange, MediaSinkListener } from './mediasink';
import Promise from './promise';
import { bufferedRange, decodedFrames, getNextPlayablePosition } from './video-element-helper';

// constants
const ERR_BUFFERING_TIMEOUT = 101;
const UNKNOWN = -1;

// configurations
export const MIN_PLAYABLE_BUFFER = 0.1; // consider buffers smaller than this as empty
const HEARTBEAT_INTERVAL = 2000; // Interval to check for stalled
const BUFFERING_TIMEOUT = 120000;

/**
 * Monitor playback and fire events accordingly
 */
export class PlaybackMonitor {
    private readonly video: HTMLVideoElement;
    private readonly listener: MediaSinkListener;
    private readonly onDelete: Array<() => void>;
    private intervalId: number;
    private idle: boolean;
    private paused: boolean;
    private lastPlayhead: number;
    private lastBufferEnd: number;
    private fps: number;
    private lastDecodedFrames: number;
    private lastTimeUpdate: number;
    private idleTimeout: number;
    private isPassThrough: boolean;
    private isWaiting: boolean;

    constructor(video: HTMLVideoElement, listener: MediaSinkListener) {
        this.video = video;
        this.listener = listener;
        this.onDelete = [];
        this.intervalId = 0;
        this.idle = true;
        this.paused = true;
        this.lastPlayhead = 0;
        this.lastBufferEnd = 0;
        this.fps = 0;
        this.lastDecodedFrames = 0;
        this.lastTimeUpdate = performance.now();
        this.idleTimeout = -1;
        this.isPassThrough = false;
        this.isWaiting = false;

        const addListener = (type, handler) => {
            this.video.addEventListener(type, handler);
            this.onDelete.push(() =>
                this.video.removeEventListener(type, handler));
        };

        addListener('play', () => this.onVideoPlay());
        addListener('pause', () => this.onVideoPause());
        addListener('durationchange', () => this.onVideoDurationChange());
        addListener('timeupdate', () => this.onVideoTimeUpdate());
        addListener('ended', () => this.onVideoEnded());
        addListener('error', () => this.onVideoError());
        addListener('waiting', () => this.onVideoWaiting());
        addListener('playing', () => this.onVideoPlaying());
    }

    /**
     * Cleanup everything so all objects can be garbage collected
     */
    delete() {
        this.onDelete.forEach((fn) => fn());
        clearInterval(this.intervalId);
    }

    /**
     * Start playback and return promise from 'video.play()'
     */
    play() {
        this.paused = false;

        // make sure playback starts within the buffer
        const buffered = this.video.buffered;
        if (buffered.length > 0) {
            const start = buffered.start(buffered.length - 1);
            const end = buffered.end(buffered.length - 1);
            if (
                // Passthrough mode, live and some buffered content behind playhead
                (
                    this.isPassThrough &&
                    this.video.duration === Infinity &&
                    end < this.video.currentTime
                ) ||
                // OR some buffered content ahead of playhead
                this.video.currentTime < start
            ) {
                console.warn('Moving to buffered region');
                this.video.currentTime = start;
            }
        }

        Promise.resolve(this.video.play()).catch(() => this.checkStopped(true));
        this.listener.onSinkPlay();
    }

    /**
     * Pause video playback
     */
    pause() {
        this.paused = true;
        this.video.pause();
    }

    /**
     * Mark that no more segments will be appended
     */
    endOfStream() {
        // We need to guarantee a final 'idle' is emitted, so we
        // don't buffer forever at the end of the stream
        this.idle = false;
        clearTimeout(this.idleTimeout);
        this.idleTimeout = -1;
    }

    /**
     * @return {number} video framerate is fps
     */
    framerate() {
        return this.fps;
    }

    updatePassThrough(isPassThrough: boolean) {
        this.isPassThrough = isPassThrough;
        if (isPassThrough && this.video.paused && !this.paused) {
            this.play();
        }
    }

    private onVideoPlay() {
        this.lastPlayhead = this.video.currentTime;
        clearInterval(this.intervalId);
        this.intervalId = self.setInterval(() => this.heartbeat(), HEARTBEAT_INTERVAL);
    }

    private onVideoPause() {
        clearInterval(this.intervalId);
        this.checkStopped(false);
    }

    private onVideoDurationChange() {
        if (this.isPassThrough) {
            this.listener.onSinkDurationChanged(this.video.duration);
        }
    }

    private onVideoTimeUpdate() {
        clearTimeout(this.idleTimeout);
        this.idleTimeout = -1;

        // BROWER BUG: Safari isn't emitting 'playing' when resuming from 'waiting'
        if (this.isPassThrough && this.isWaiting && this.video.readyState === 4) {
            this.isWaiting = false;
            this.listener.onSinkPlaying(this.paused);
        }

        // Update framerate
        const decoded = decodedFrames(this.video);
        const now = performance.now();
        this.fps = 1000 * Math.max(decoded - this.lastDecodedFrames, 0) / (now - this.lastTimeUpdate);
        this.lastDecodedFrames = decoded;
        this.lastTimeUpdate = now;

        this.listener.onSinkTimeUpdate();
        const buffered = bufferedRange(this.video, MIN_PLAYABLE_BUFFER);
        this.checkBufferUpdate(buffered);

        if (!this.isPassThrough) {
            this.updateIdle(buffered);
        }
    }

    private onVideoEnded() {
        this.listener.onSinkEnded();
    }

    private onVideoWaiting() {
        this.isWaiting = true;
        if (this.isPassThrough && !this.video.seeking) {
            this.listener.onSinkIdle();
        }
    }

    private onVideoPlaying() {
        this.isWaiting = false;
        this.listener.onSinkPlaying(this.paused);
    }

    private onVideoError() {
        const { code, message = '' } = this.video.error;
        this.listener.onSinkError({
            value: code,
            code,
            message,
        });
    }

    /**
     * A heartbeat that monitors playback progress and corrects any playback issues.
     */
    private heartbeat() {
        const buffered = bufferedRange(this.video, MIN_PLAYABLE_BUFFER);
        if (this.video.paused) {
            // This shouldn't happen, but stop heartbeat
            // if we get into a bad state.
            clearInterval(this.intervalId);
        } else if (this.video.currentTime === this.lastPlayhead) {
            const nextPlayablePosition = getNextPlayablePosition(this.video, MIN_PLAYABLE_BUFFER);
            if (nextPlayablePosition !== this.video.currentTime) {
                console.warn(`jumping ${nextPlayablePosition - this.video.currentTime}s gap`);
                this.video.currentTime = nextPlayablePosition;
            }

            this.updateIdle(buffered);
        } else {
            this.checkBufferUpdate(buffered);
            this.lastPlayhead = this.video.currentTime;
        }

        this.videoDisplaySizeUpdate();
    }

    private videoDisplaySizeUpdate() {
        const w = this.video.clientWidth * window.devicePixelRatio;
        const h = this.video.clientHeight * window.devicePixelRatio;
        this.listener.onSinkVideoDisplaySizeChanged(w, h);
    }

    private checkBufferUpdate({ end }: BufferRange) {
        if (end !== this.lastBufferEnd) {
            this.lastBufferEnd = end;
            this.listener.onSinkBufferUpdate();
        }
    }

    /**
     * Update our idle state, and emit 'idle' if we're transitioning too idle.
     * @param  {Object} buffered - current buffered range
     */
    private updateIdle({ start, end }: BufferRange) {
        if (this.video.paused) {
            this.idle = true;
        } else {
            const remaining = end - this.video.currentTime;
            const ended = this.video.ended || (this.video.duration - this.video.currentTime < MIN_PLAYABLE_BUFFER);
            const idle = !ended && (remaining < MIN_PLAYABLE_BUFFER);
            if (idle && !this.idle) {
                this.idleTimeout = self.setTimeout(() => this.onBufferingTimeout(), BUFFERING_TIMEOUT);
                this.listener.onSinkIdle();
            }
            this.idle = idle;
        }
    }

    /**
     * Buffering timeout handler. Emit onError with error codes for buffering timeout
     */
    private onBufferingTimeout() {
        clearTimeout(this.idleTimeout);
        this.idleTimeout = -1;
        this.listener.onSinkError({
            value: ERR_BUFFERING_TIMEOUT,
            code: ERR_BUFFERING_TIMEOUT,
            message: 'Buffering timeout',
        });
    }

    /**
     * Notify mediaplayer if the browser paused on its own. This can happen
     * when the player is muted and hidden.
     */
    private checkStopped(playFailed: boolean) {
        if (this.video.paused && !this.video.ended && !this.video.error && !this.paused) { // An auto pause scenario
            this.listener.onSinkStop(playFailed);
        }
    }
}
