import { Quality } from './web-mediaplayer';

/**
 * Interface for controlling a Player instance. API users instantiate a player implementation
 * and issue commands to it.
 */
export interface Player {
    /**
     * Loads the given stream.
     *
     * @param path of the stream to load (e.g. a url or file)
     * @param mediaType Media type of the content if known, e.g. 'video/mp4' or 'application/x-mpegURL'
     */
    load(path: string, mediaType: string): void;

    /**
     * Starts or resumes playback of the current stream or fails if no stream is loaded.
     * On success depending on the type of stream the player state will change to BUFFERING
     * and then PlayerState.PLAYING or just PlayerState.PLAYING. On failure invokes the onError listener
     * method.
     */
    play(): void;

    /**
     * Enables or disables autoplaying.
     * @param enabled true to set autoplay, false otherwise
     */
    setAutoplay(enabled: boolean): void;

    /**
     * Pauses playback of the current stream or fails if no stream is loaded. On failure invokes the
     * onError listener method.
     *
     * Resuming the stream with play may result in the position being different depending on the
     * type of media being played.
     */
    pause(): void;

    /**
     * Seeks to the given time in the stream and begins playing at that position if play() has been
     * called. On success depending on the type of stream the player state will change to
     * PlayerState.BUFFERING and then PlayerState.PLAYING or remain in PlayerState.PLAYING. getPosition() will
     * update to the seeked time. On failure invokes the onError listener method.
     *
     * @param time to seek to
     */
    seekTo(time: number): void;

    /**
     * Enables the player to loop the source video if supported
     * @param loop true to loop video playback, false otherwise
     */
    setLooping(loop: boolean): void;

    /**
     * Mutes or unmutes the player
     * @param mute true to mute the stream false to unmute
     */
    setMuted(mute: boolean): void;

    /**
     * Sets the playback volume of the audio track for the player (either for the current stream or
     * the next stream that is loaded).
     * @param volume value to set between 0.0f and 1.0f
     */
    setVolume(volume: number): void;

    /**
     * Sets the current quality of the stream, if auto switch quality it is disabled.
     * @param quality a valid quality entry from getQualities()
     * @param adaptive, true for an adaptive quality change (quality changes at the end
     * of the buffer, false otherwise buffer cleared immediately)
     */
    setQuality(quality: Quality, adaptive: boolean): void;

    setAuthToken(token: string): void;

    /** @return true if automatic quality switching is enabled, false otherwise */
    getAutoSwitchQuality(): boolean;

    /** @return the current quality of the source */
    getQuality(): Quality;

    /** @return the available qualities or empty if none are currently available */
    getQualities(): Quality[];

    /**
     * Enables automatic quality selection (based on the device/network) when possible
     * @param enable true to enable, false to disable
     */
    setAutoSwitchQuality(enable: boolean): void;

    /**
     * Sets the initial starting bitrate when using auto quality switching. This determines the
     * initial quality an adaptive stream will playback at when no bandwidth estimate value has
     * been determined.
     * @param bitrate initial bitrate to use
     */
    setAutoInitialBitrate(bitrate: number): void;
    /**
     * Sets the max bitrate to use when using auto quality switching. Allows controlling resource
     * usage in the case of multiple players.
     * @param bitrate initial bitrate to use
     */
    setAutoMaxBitrate(bitrate: number): void;

    /**
     * Sets the max video display size of the player, which prevents switching to qualities above
     * this resolution when autoSwitchQuality is true.
     * @param width display width
     * @param height display height
     */
    setAutoMaxVideoSize(width: number, height: number): void;

    setAutoViewportSize(width: number, height: number): void;
    setAutoViewportScale(scale: number): void;
    setAutoViewportSizeEnabled(enable: boolean): void;

    /**
     * Sets the current playback rate of the stream.
     * @param rate playback rate to set
     */
    setPlaybackRate(rate: number): void;

    /**
     * Sets the API client id used for internal requests made by the player
     * @param id to set
     */
    setClientId(id: string): void;

    /**
     * Sets the device id for this player instance used for API requests
     * @param id to set
     */
    setDeviceId(id: string): void;

    /**
     * Sets speedup rate used by player
     */
    setLiveSpeedUpRate(rate: number): void;

    /**
     * Sets player type
     * @param {string} type (e.g. 'site' or 'embed')
     */
    setPlayerType(type: string): void;

    /**
     * Sets a target live latency, during buffering events if the latency would exceed this value
     * the live stream will be re-syncd to the live window to prevent the latency exceeding this
     * value.
     * @param {number} latency max latency before attempting to skip content during live playback
     */
    setLiveMaxLatency(latency: number): void;

    /**
     * Enables low latency playback for streams which support it. Defaults to true, changing the
     * value during playback will restart the stream.
     *
     * @param {boolean} enable true to enable, false to disable
     */
    setLiveLowLatencyEnabled(enable: boolean): void;

    /**
     * Sets the minimum buffer size needed to start playback.
     * @param {number} duration duration of buffer to set in seconds
     */
    setMinBuffer(duration: number): void;

    /**
     * Sets the target buffer size to buffer ahead of the playhead.
     * @param {number} duration duration of buffer to set in seconds
     */
    setMaxBuffer(duration: number): void;

    /**
     * Sets the visibility hint, if not visible less resources will be used.
     * @param {boolean} visible true if the player is visible on screen, false otherwise
     */
    setVisible(visible: boolean): void;

    /**
     * @deprecated
     */
    setAnalyticsEndpoint(url: string): void;

    /**
     * @deprecated
     */
    setAnalyticsSendEvents(enable: boolean): void;

    /**
     * @deprecated
     */
    setBufferReplaceEnabled(enable: boolean): void;

    addEventListener<T>(name: string, callback: (obj: T) => void): void;

    getPlayerState(): String;
}
