import { StreamReaderShim } from './stream-reader-shim';

export type StreamingResponseType = XMLHttpRequest['responseType'] |
                                    'moz-chunked-arraybuffer' |
                                    'ms-stream';

// Basic version of https://streams.spec.whatwg.org/
export class ReadableStreamShim {
    private reader: StreamReaderShim;

    constructor(xhr: XMLHttpRequest, responseType: StreamingResponseType) {
        this.reader = new StreamReaderShim(xhr.abort.bind(xhr));
        this.initReadableStreamShim(xhr, responseType);
    }

    getReader(): StreamReaderShim {
        return this.reader;
    }

    // tslint:disable-next-line: no-any
    private initReadableStreamShim(xhr: XMLHttpRequest, responseType: StreamingResponseType) {
        xhr.responseType = responseType as XMLHttpRequest['responseType'];

        // set up progress listeners based on response type
        switch (responseType) {
        case 'moz-chunked-arraybuffer':
            xhr.addEventListener('progress', () => {
                this.reader.write(new Uint8Array(xhr.response));
            });
            xhr.addEventListener('load', this.reader.close.bind(this.reader));
            break;
        case 'ms-stream':
            xhr.addEventListener('readystatechange', () => {
                // Only run once when switching to LOADING
                if (xhr.readyState !== xhr.LOADING) {
                    return;
                }

                const msReader = new self['MSStreamReader']();
                let offset = 0;
                msReader.onprogress = () => {
                    if (msReader.result.byteLength > offset) {
                        this.reader.write(new Uint8Array(msReader.result, offset));
                        offset = msReader.result.byteLength;
                    }
                };
                msReader.onload = this.reader.close.bind(this.reader);
                msReader.readAsArrayBuffer(xhr.response);
            });
            break;
        case 'arraybuffer':
            // add zero-sized chunks so we're updated on download progress
            // Can bind since same value each time
            xhr.addEventListener('progress', this.reader.write.bind(this.reader, new Uint8Array(0)));
            xhr.addEventListener('load', () => {
                if (xhr.response) {
                    this.reader.write(new Uint8Array(xhr.response));
                }
                this.reader.close();
            });
            break;
        default:
            break;
        }
    }
}
