import { SafeSourceBuffer } from './safe-source-buffer';
import {
    addMissingVideoElementProperties,
    createMockSourceBuffer,
    createMockTimeRanges,
    fireUpdateEndListenerOnSourceBuffer,
} from './test-utils/video-element-api';

beforeAll(() => {
    addMissingVideoElementProperties();
});

function setupSafeSourceBuffer(updatingMode: boolean = false) {
    const videoElement = document.createElement('video');
    // tslint:disable-next-line no-any
    const sourceBuffer: any = createMockSourceBuffer(updatingMode);
    const specimenSafeSourceBuffer = new SafeSourceBuffer(videoElement, sourceBuffer);

    return { videoElement, sourceBuffer, specimenSafeSourceBuffer };
}

describe('Test basic APIs of SafeSourceBuffer when it is not updating', () => {
    test('Test appendBuffer', () => {
        const { sourceBuffer, specimenSafeSourceBuffer } = setupSafeSourceBuffer();
        const spy = jest.spyOn(sourceBuffer, 'appendBuffer');

        const specimenBuffer = new ArrayBuffer(8);
        specimenSafeSourceBuffer.appendBuffer(specimenBuffer);
        expect(spy).toHaveBeenCalledWith(specimenBuffer);
    });

    test('Test setTimestampOffset', () => {
        const { sourceBuffer, specimenSafeSourceBuffer } = setupSafeSourceBuffer();
        const sampleValue = 5;
        specimenSafeSourceBuffer.setTimestampOffset(sampleValue);
        expect(sourceBuffer.timestampOffset).toEqual(sampleValue);
    });

    test('Test remove', () => {
        const { sourceBuffer, specimenSafeSourceBuffer } = setupSafeSourceBuffer();

        // Mock buffered time range
        const bufferedMinStartTime = 2.23;
        const bufferedMaxEndTime = 15.52;
        createMockTimeRanges(sourceBuffer, 2, bufferedMinStartTime, bufferedMaxEndTime);

        const spy = jest.spyOn(sourceBuffer, 'remove');

        const specimentStartTime = 1.5;
        const specimenEndTime = 12.23;
        specimenSafeSourceBuffer.remove(specimentStartTime, specimenEndTime);

        expect(spy).toBeCalledWith(Math.max(specimentStartTime, bufferedMinStartTime), Math.min(specimenEndTime, bufferedMaxEndTime));
    });
});

describe('Test basic APIs of SafeSourceBuffer when it is updating', () => {
    test('Test single appendBuffer', () => {
        const { sourceBuffer, specimenSafeSourceBuffer } = setupSafeSourceBuffer(true);
        const spy = jest.spyOn(sourceBuffer, 'appendBuffer');

        const specimenBuffer = new ArrayBuffer(8);
        specimenSafeSourceBuffer.appendBuffer(specimenBuffer);
        expect(spy).not.toBeCalled();

        fireUpdateEndListenerOnSourceBuffer(sourceBuffer);
        expect(spy).toHaveBeenCalledWith(specimenBuffer);
    });

    test('Test multiple appendBuffer', () => {
        const { sourceBuffer, specimenSafeSourceBuffer } = setupSafeSourceBuffer(true);
        const spy = jest.spyOn(sourceBuffer, 'appendBuffer');

        const specimenBuffer1 = new ArrayBuffer(8);
        specimenSafeSourceBuffer.appendBuffer(specimenBuffer1);
        const specimenBuffer2 = new ArrayBuffer(16);
        specimenSafeSourceBuffer.appendBuffer(specimenBuffer2);
        const specimenBuffer3 = new ArrayBuffer(24);
        specimenSafeSourceBuffer.appendBuffer(specimenBuffer3);

        expect(spy).not.toBeCalled();

        fireUpdateEndListenerOnSourceBuffer(sourceBuffer);

        // Calls [0] -> Array of arguments of the first invocation
        expect(spy.mock.calls[0][0]).toEqual(specimenBuffer1);
        expect(spy.mock.calls[1][0]).toEqual(specimenBuffer2);
        expect(spy.mock.calls[2][0]).toEqual(specimenBuffer3);
    });

    test('Test combination of APIs appendBuffer, setTimestampOffset and remove', () => {
        const { sourceBuffer, specimenSafeSourceBuffer } = setupSafeSourceBuffer(true);
        const spyOnAppendBuffer = jest.spyOn(sourceBuffer, 'appendBuffer');
        const spyOnSetTimestampOffset = jest.spyOn(sourceBuffer, 'timestampOffset', 'set');
        const spyOnRemove = jest.spyOn(sourceBuffer, 'remove');

        // 1. Append
        specimenSafeSourceBuffer.appendBuffer(new ArrayBuffer(8));
        // 2. setTimestampOffset
        specimenSafeSourceBuffer.setTimestampOffset(5);
        // 3. Append
        specimenSafeSourceBuffer.appendBuffer(new ArrayBuffer(16));
        // 4. Remove
        createMockTimeRanges(sourceBuffer, 2, 2.23, 15.52);
        specimenSafeSourceBuffer.remove(10, 20);
        // 5. setTimestampOffset
        specimenSafeSourceBuffer.setTimestampOffset(6);

        // Expect not being called before updateend
        expect(spyOnAppendBuffer).not.toBeCalled();
        expect(spyOnSetTimestampOffset).not.toBeCalled();
        expect(spyOnRemove).not.toBeCalled();

        fireUpdateEndListenerOnSourceBuffer(sourceBuffer);

        // Expect the number of calls
        expect(spyOnAppendBuffer).toBeCalledTimes(2);
        expect(spyOnSetTimestampOffset).toBeCalledTimes(2);
        expect(spyOnRemove).toBeCalledTimes(1);

        const invocationCallOrderOfAppendBuffer = spyOnAppendBuffer.mock.invocationCallOrder;
        const invocationCallOrderOfSetTimestampOffset = spyOnSetTimestampOffset.mock.invocationCallOrder;
        const invocationCallOrderOfRemove = spyOnRemove.mock.invocationCallOrder;

        // Expect the invocation order to be correct
        expect(invocationCallOrderOfAppendBuffer[0]).toBeLessThan(invocationCallOrderOfSetTimestampOffset[0]);
        expect(invocationCallOrderOfSetTimestampOffset[0]).toBeLessThan(invocationCallOrderOfAppendBuffer[1]);
        expect(invocationCallOrderOfAppendBuffer[1]).toBeLessThan(invocationCallOrderOfRemove[0]);
        expect(invocationCallOrderOfRemove[0]).toBeLessThan(invocationCallOrderOfSetTimestampOffset[1]);
    });
});
