import { Queue } from './queue';

/**
 * SafeSourceBuffer wraps a SourceBuffer, and schedules
 * 'appendBuffer' and 'remove' operations correctly
 */
export class SafeSourceBuffer {
    private video: HTMLVideoElement;
    private srcBuf: SourceBuffer;
    private pending: Queue<(srcBuf: SourceBuffer) => void>;

    constructor(videoElement: HTMLVideoElement, srcBuf: SourceBuffer) {
        this.video = videoElement;
        this.srcBuf = srcBuf;
        this.pending = new Queue();
        this.srcBuf.addEventListener('updateend', this.process.bind(this));
    }

    /**
     * Append a media buffer
     * @param {ArrayBuffer} buf - fmp4 media buffer
     */
    appendBuffer(buf: ArrayBuffer) {
        this.schedule((sourceBuffer) => {
            try {
                sourceBuffer.appendBuffer(buf);
            } catch (e) {
                if (e.name === 'QuotaExceededError') {
                    const { buffered } = sourceBuffer;
                    const start = buffered.start(0);
                    const end = buffered.end(buffered.length - 1);
                    const mid = (start + end) / 2;
                    sourceBuffer.remove(mid, end);
                }
            }
        });
    }

    /**
     * Update timestamp offset
     * @param {number} offset - offset to apply in seconds
     */
    setTimestampOffset(offset: number) {
        this.schedule((sourceBuffer) => {
            sourceBuffer.timestampOffset = offset;
        });
    }

    /**
     * Remove range of buffered media
     * @param {number} start - start of range to remove
     * @param {number} end - end of range to remove
     */
    remove(start: number, end: number) {
        this.schedule((sourceBuffer) => {
            const { buffered } = sourceBuffer;
            if (buffered.length) {
                // Edge doesn't like large remove ranges, so we'll
                // clamp remove calls to the range of the current buffer.
                // 'start' can become greater than 'end' if the remove
                // range doesn't contain the current buffer.
                const maxStart = Math.max(start, buffered.start(0));
                const minEnd = Math.min(end, buffered.end(buffered.length - 1));
                if (maxStart < minEnd) {
                    sourceBuffer.remove(maxStart, minEnd);
                }
            }
        });
    }

    /**
     * Schedule a callback for when updating is over
     * @param {fn} callback - call after update ends
     */
    private schedule(fn: (srcBuf: SourceBuffer) => void) {
        if (this.pending.empty() && !this.updating()) {
            fn(this.srcBuf);
        } else {
            this.pending.push(fn);
            this.process();
        }
    }

    /**
     * Try to process as many buffer updates as we can
     */
    private process() {
        while (!this.pending.empty() && !this.updating()) {
            this.pending.pop()(this.srcBuf);
        }
    }

    private updating(): boolean {
        return (!this.srcBuf || this.srcBuf.updating || this.video.error !== null);
    }
}
