import { SafeTextTrack } from './safe-text-track';
import { addMissingTextTrackImplementation, addMissingVTTCueImplementation } from './test-utils/video-element-api';

beforeAll(() => {
    addMissingTextTrackImplementation();
    addMissingVTTCueImplementation();
});

describe('Test basic APIs of SafeTextTrack', () => {
    test('Test addition of non-overlapping cues to video element\'s text track uses same texttrack', () => {
        const videoElement = document.createElement('video');
        const specimenSafeTextTrack = new SafeTextTrack(videoElement);

        const sampleCue1 = new VTTCue(0, 10, 'Sample cue from 0s to 10s');
        specimenSafeTextTrack.addCue(sampleCue1);
        expect(videoElement.textTracks.length).toEqual(1);

        // Add another non overlapping cue
        const sampleCue2 = new VTTCue(20, 30, 'Sample cue from 20s to 30s');
        specimenSafeTextTrack.addCue(sampleCue2);
        expect(videoElement.textTracks.length).toEqual(1);
    });

    test('Test addition of overlapping cues to video element\'s text track, results in separate texttrack', () => {
        const videoElement = document.createElement('video');
        const specimenSafeTextTrack = new SafeTextTrack(videoElement);

        const sampleCue1 = new VTTCue(0, 10, 'Sample cue from 0s to 10s');
        specimenSafeTextTrack.addCue(sampleCue1);
        expect(videoElement.textTracks.length).toEqual(1);

        // Add another overlapping cue
        const sampleCue2 = new VTTCue(4, 6, 'Sample cue from 4s to 6s');
        specimenSafeTextTrack.addCue(sampleCue2);
        expect(videoElement.textTracks.length).toEqual(1);
    });

    test('Test removal of cues from the text tracks', () => {
        const videoElement = document.createElement('video');
        const specimenSafeTextTrack = new SafeTextTrack(videoElement);

        specimenSafeTextTrack.addCue(new VTTCue(0, 10, 'Sample cue from 0s to 10s'));
        specimenSafeTextTrack.addCue(new VTTCue(11, 15, 'Sample cue from 11s to 15s'));
        // Getting reference to the cue expected to be removed
        const sampleCue = new VTTCue(4, 6, 'Sample cue from 4s to 6s');
        specimenSafeTextTrack.addCue(sampleCue);
        specimenSafeTextTrack.addCue(new VTTCue(2, 12, 'Sample cue from 2s to 12s'));

        const textTracks = videoElement.textTracks;
        expect(textTracks.length).toEqual(3); // Text tracks should avoid overlaps

        const spyOnTrack1 = jest.spyOn(textTracks[0], 'removeCue');
        const spyOnTrack2 = jest.spyOn(textTracks[1], 'removeCue');
        const spyOnTrack3 = jest.spyOn(textTracks[2], 'removeCue');

        specimenSafeTextTrack.remove(4, 14);

        expect(textTracks.length).toEqual(3); // Text tracks are not cleared

        expect(spyOnTrack1).not.toBeCalled();
        expect(spyOnTrack2).toBeCalledWith(sampleCue); // (4,6) is within removal window of (4,14)
        expect(spyOnTrack3).not.toBeCalled();
    });
});
