/** SafeTextTrack
 * There's an issue on Edge where text track cues must be added in accending order
 * To handle this, we create a new text track every time we need to add an out-of-order cue
 */

export class SafeTextTrack {
    private readonly video: HTMLVideoElement;
    private readonly tracks: TextTrack[];

    constructor(videoElement: HTMLVideoElement) {
        this.video = videoElement;
        this.tracks = [videoElement.addTextTrack('metadata')];
    }

    /**
     * Add a VTTCue to the virtual text track
     * @param {VTTCue} cue - The cue to add
     */
    addCue(cue: VTTCue) {
        let validTrack: TextTrack;

        for (const track of this.tracks) {
            const { cues } = track;
            if (
                !cues.length ||
                (
                    /**
                     * MS Edge has issues with cues added out of order
                     * First discovered - https://jira.twitch.com/browse/CVP-2544
                     * Rediscovered - https://jira.twitch.com/browse/VP-6188
                     * New cue cannot start before last one
                     * New cue cannot end same as last one
                     */
                    cues[cues.length - 1].startTime < cue.startTime &&
                    cues[cues.length - 1].endTime !== cue.endTime
                )
            ) {
                validTrack = track;
                break;
            }
        }

        if (!validTrack) {
            validTrack = this.video.addTextTrack('metadata');
            this.tracks.push(validTrack);
        }

        try {
            validTrack.addCue(cue);
        } catch (e) {
            const newTrack = this.video.addTextTrack('metadata');
            this.tracks.push(newTrack);
            try {
                newTrack.addCue(cue);
            } catch (e) {
                console.warn('Player could not add cue at start ', cue.startTime, 'end ', cue.endTime);
            }
        }
    }

    /**
     * Remove any cues in the range from [start, end)
     * @param {Number} start - start in seconds
     * @param {Number} end - end in seconds
     */
    remove(start: number, end: number) {
        this.tracks.forEach((track) => {
            const cues = track.cues;
            for (let index = cues.length - 1; index >= 0; index--) {
                const cue = cues[index];

                if (cue.startTime < start) { break; }

                if (cue.endTime < end) {
                    track.removeCue(cue);
                }
            }
        });
    }
}
