import Promise from './promise';
import { Queue } from './queue';

interface ReadRequest {
    resolve: (value: ReadableStreamReadResult<Uint8Array>) => void;
    reject: (error: Error) => void;
}

const enum ReaderState {
    Readable,
    Closed,
    Errored,
}

// Basic version 'ReadableStreamDefaultReader'
// from https://streams.spec.whatwg.org/
export class StreamReaderShim {
    // Marks the end of stream
    private static readonly DONE_CHUNK = { done: true, value: undefined };

    private state: ReaderState;
    private queuedChunks: Queue<Promise<ReadableStreamReadResult<Uint8Array>>>;
    private readRequest: ReadRequest| null;
    private storedError: Error | null;
    private onCancel: () => void;

    constructor(onCancel: () => void) {
        this.state = ReaderState.Readable;
        this.queuedChunks = new Queue();
        this.readRequest = null;
        this.storedError = null;
        this.onCancel = onCancel;
    }

    // Calling `read` while there is an outstanding
    // Promise from this function is undefined
    read(): Promise<ReadableStreamReadResult<Uint8Array>> {
        switch (this.state) {
        case ReaderState.Readable:
            // We already have data, return it immediatly
            if (!this.queuedChunks.empty()) {
                return this.queuedChunks.pop();
            }
            // create a read request if we have no more buffered data
            return new Promise((resolve, reject) => {
                this.readRequest = { resolve, reject };
            });

        case ReaderState.Closed:
            // Return any data that was written before we closed
            if (!this.queuedChunks.empty()) {
                return this.queuedChunks.pop();
            }
            return Promise.resolve(StreamReaderShim.DONE_CHUNK);

        case ReaderState.Errored:
            return Promise.reject(this.storedError);
        default:
            break;
        }
    }

    cancel() {
        this.onCancel();
        this.close();
    }

    error(err: Error) {
        switch (this.state) {
        case ReaderState.Readable:
            this.state = ReaderState.Errored;
            this.storedError = err;
            this.queuedChunks = null;
            if (this.readRequest) {
                this.readRequest.reject(err);
                this.readRequest = null;
            }
            break;
        default:
            break;
        }
    }

    write(data: Uint8Array) {
        switch (this.state) {
        case ReaderState.Readable:
            const chunk = { done: false, value: data };

            if (this.readRequest) {
                this.readRequest.resolve(chunk);
                this.readRequest = null;
            } else {
                this.queuedChunks.push(Promise.resolve(chunk));
            }
            break;
        default:
            break;
        }
    }

    close() {
        switch (this.state) {
        case ReaderState.Readable:
            if (this.readRequest) {
                this.readRequest.resolve(StreamReaderShim.DONE_CHUNK);
                this.readRequest = null;
            }
            this.state = ReaderState.Closed;
            break;
        default:
            break;
        }
    }
}
