import {
    bufferedRange,
    decodedFrames,
    droppedFrames,
    getNextPlayablePosition,
    MIN_PLAYABLE_BUFFER,
    subscribe,
    willBuffer,
} from './video-element-helper';

describe('video-element-helper functions', () => {
    describe('decodedFrames', () => {
        test('webkitDecodedFrameCount', () => {
            // Setup
            const videoElement = document.createElement('video');
            const SAMPLE_DECODED_FRAME = 100;
            // Test
            Object.defineProperty(videoElement, 'webkitDecodedFrameCount', {
                value: SAMPLE_DECODED_FRAME,
            });
            expect(decodedFrames(videoElement)).toEqual(SAMPLE_DECODED_FRAME);
        });

        test('getVideoPlaybackQuality().totalVideoFrames', () => {
            // Setup
            const videoElement = document.createElement('video');
            const SAMPLE_DECODED_FRAME = 100;

            videoElement.getVideoPlaybackQuality = () => {
                return {
                    totalVideoFrames: SAMPLE_DECODED_FRAME,
                } as VideoPlaybackQuality;
            };
            // Test
            expect(decodedFrames(videoElement)).toEqual(SAMPLE_DECODED_FRAME);
        });

        test('mozDecodedFrames', () => {
            // Setup
            const videoElement = document.createElement('video');
            const SAMPLE_DECODED_FRAME = 100;
            // Test
            Object.defineProperty(videoElement, 'mozDecodedFrames', {
                value: SAMPLE_DECODED_FRAME,
            });
            expect(decodedFrames(videoElement)).toEqual(SAMPLE_DECODED_FRAME);
        });

        test('decoded frames unavailable', () => {
            // Setup
            const videoElement = document.createElement('video');
            // Test
            expect(decodedFrames(videoElement)).toEqual(-1);
        });
    });

    describe('droppedFrames', () => {
        test('webkitDroppedFrameCount', () => {
            // Setup
            const videoElement = document.createElement('video');
            const SAMPLE_DROPPED_FRAMES = 20;
            // Test
            Object.defineProperty(videoElement, 'webkitDroppedFrameCount', {
                value: SAMPLE_DROPPED_FRAMES,
            });
            expect(droppedFrames(videoElement)).toEqual(SAMPLE_DROPPED_FRAMES);
        });

        test('getVideoPlaybackQuality().droppedVideoFrames', () => {
            // Setup
            const videoElement = document.createElement('video');
            const SAMPLE_DROPPED_FRAMES = 20;

            videoElement.getVideoPlaybackQuality = () => {
                return {
                    droppedVideoFrames: SAMPLE_DROPPED_FRAMES,
                } as VideoPlaybackQuality;
            };
            // Test
            expect(droppedFrames(videoElement)).toEqual(SAMPLE_DROPPED_FRAMES);
        });

        test('dropped frames unavailable', () => {
            // Setup
            const videoElement = document.createElement('video');
            // Test
            expect(droppedFrames(videoElement)).toEqual(-1);
        });
    });

    describe('bufferedRange', () => {
        test('playhead clearly inside buffered region', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead inside the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 120;
            const CURRENT_PLAYHEAD = 100;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });

            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).start).toEqual(BUFFER_START_POSITION);
            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).end).toEqual(BUFFER_END_POSITION);
        });

        test('playhead outside of buffered region', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead past the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 120;
            const CURRENT_PLAYHEAD = 40;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });

            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).start).toEqual(CURRENT_PLAYHEAD);
            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).end).toEqual(CURRENT_PLAYHEAD);
        });

        test('playhead close to buffered region within MIN_PLAYABLE_BUFFER', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead past the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 120;
            const CURRENT_PLAYHEAD = BUFFER_START_POSITION - MIN_PLAYABLE_BUFFER;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });

            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).start).toEqual(CURRENT_PLAYHEAD);
            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).end).toEqual(BUFFER_END_POSITION);
        });

        test('playhead part of a region-1, but there is another region-2 AHEAD close by MIN_PLAYABLE_BUFFER. Should consider that region as well', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead past the buffered region
            const BUFFER_START_POSITION_2 = 50;
            const BUFFER_END_POSITION_2 = BUFFER_START_POSITION_2 + (0.988 * MIN_PLAYABLE_BUFFER);

            const BUFFER_START_POSITION_1 = 40;
            const BUFFER_END_POSITION_1 = BUFFER_START_POSITION_2 - (0.95 * MIN_PLAYABLE_BUFFER);
            const CURRENT_PLAYHEAD = BUFFER_END_POSITION_1 - MIN_PLAYABLE_BUFFER;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            const starts = [BUFFER_START_POSITION_1, BUFFER_START_POSITION_2];
            const ends = [BUFFER_END_POSITION_1, BUFFER_END_POSITION_2];
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: starts.length,
                start: (i) => {
                    return starts[i];
                },
                end: (i) => {
                    return ends[i];
                },
            });

            // Test
            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).start).toEqual(BUFFER_START_POSITION_1);
            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).end).toEqual(BUFFER_END_POSITION_2);
        });

        test('playhead part of a region-2, but there is another region-1 BEHIND close by MIN_PLAYABLE_BUFFER. Should consider that region as well', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead past the buffered region
            const BUFFER_START_POSITION_2 = 50;
            const BUFFER_END_POSITION_2 = BUFFER_START_POSITION_2 + (0.988 * MIN_PLAYABLE_BUFFER);

            const BUFFER_START_POSITION_1 = 40;
            const BUFFER_END_POSITION_1 = BUFFER_START_POSITION_2 - (0.95 * MIN_PLAYABLE_BUFFER);

            const CURRENT_PLAYHEAD = BUFFER_START_POSITION_2 + (0.93 * MIN_PLAYABLE_BUFFER);

            videoElement.currentTime = CURRENT_PLAYHEAD;
            const starts = [BUFFER_START_POSITION_1, BUFFER_START_POSITION_2];
            const ends = [BUFFER_END_POSITION_1, BUFFER_END_POSITION_2];
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: starts.length,
                start: (i) => {
                    return starts[i];
                },
                end: (i) => {
                    return ends[i];
                },
            });

            // Test
            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).start).toEqual(BUFFER_START_POSITION_1);
            expect(bufferedRange(videoElement, MIN_PLAYABLE_BUFFER).end).toEqual(BUFFER_END_POSITION_2);
        });
    });

    describe('willBuffer', () => {
        test('if playhead has enough buffer ahead return false', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead inside the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 120;
            const DURATION = 150;
            const CURRENT_PLAYHEAD = 100;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            Object.defineProperty(videoElement, 'duration', {
                value: DURATION,
            });
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });

            // Test
            expect(willBuffer(videoElement, MIN_PLAYABLE_BUFFER)).toEqual(false);
        });

        test('if playhead is close to buffer end return true', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead inside the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 120;
            const DURATION = 150;
            const CURRENT_PLAYHEAD = BUFFER_END_POSITION - (0.95 * MIN_PLAYABLE_BUFFER);

            videoElement.currentTime = CURRENT_PLAYHEAD;
            Object.defineProperty(videoElement, 'duration', {
                value: DURATION,
            });
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });

            // Test
            expect(willBuffer(videoElement, MIN_PLAYABLE_BUFFER)).toEqual(true);
        });

        test('if playhead is close to duration return false', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead inside the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 120;
            const CURRENT_PLAYHEAD = BUFFER_END_POSITION - (0.95 * MIN_PLAYABLE_BUFFER);

            videoElement.currentTime = CURRENT_PLAYHEAD;
            Object.defineProperty(videoElement, 'duration', {
                value: BUFFER_END_POSITION,
            });
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });

            // Test
            expect(willBuffer(videoElement, MIN_PLAYABLE_BUFFER)).toEqual(false);
        });

        test('If video has ended return false', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead inside the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 120;
            const CURRENT_PLAYHEAD = BUFFER_END_POSITION;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            Object.defineProperty(videoElement, 'ended', {
                value: true,
            });
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });

            // Test
            expect(willBuffer(videoElement, MIN_PLAYABLE_BUFFER)).toEqual(false);
        });
    });

    describe('getNextPlayablePosition', () => {
        test('playhead is inside the buffer region expect a nudge of +MIN_PLAYABLE_BUFFER', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead inside the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 120;
            const CURRENT_PLAYHEAD = 100;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });

            // Test
            expect(getNextPlayablePosition(videoElement, MIN_PLAYABLE_BUFFER)).toEqual(CURRENT_PLAYHEAD + MIN_PLAYABLE_BUFFER);
        });

        test('buffer region ahead of playhead expect the return value to be that regions\'s start + MIN_PLAYABLE_BUFFER ', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead inside the buffered region
            const BUFFER_START_POSITION = 50;
            const BUFFER_END_POSITION = 120;
            const CURRENT_PLAYHEAD = 30;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: 1,
                start: jest.fn().mockReturnValue(BUFFER_START_POSITION),
                end: jest.fn().mockReturnValue(BUFFER_END_POSITION),
            });

            // Test
            expect(getNextPlayablePosition(videoElement, MIN_PLAYABLE_BUFFER)).toEqual(BUFFER_START_POSITION + MIN_PLAYABLE_BUFFER);
        });

        test('buffer regions ahead of playhead some not larger than MIN_PLAYABLE_BUFFER, expect move to the larger region', () => {
            const videoElement = document.createElement('video');

            // Setup video element to have the playhead past the buffered region
            const BUFFER_START_POSITION_1 = 10;
            const BUFFER_END_POSITION_1 = BUFFER_START_POSITION_1 + (0.988 * MIN_PLAYABLE_BUFFER);

            const BUFFER_START_POSITION_2 = 40;
            const BUFFER_END_POSITION_2 = 50;

            const CURRENT_PLAYHEAD = 5;

            videoElement.currentTime = CURRENT_PLAYHEAD;
            const starts = [BUFFER_START_POSITION_1, BUFFER_START_POSITION_2];
            const ends = [BUFFER_END_POSITION_1, BUFFER_END_POSITION_2];
            jest.spyOn(videoElement, 'buffered', 'get').mockReturnValue({
                length: starts.length,
                start: (i) => {
                    return starts[i];
                },
                end: (i) => {
                    return ends[i];
                },
            });

            // Test
            expect(getNextPlayablePosition(videoElement, MIN_PLAYABLE_BUFFER)).toEqual(BUFFER_START_POSITION_2 + MIN_PLAYABLE_BUFFER);
        });
    });

    test('subscribe', () => {
        const videoElement = document.createElement('video');
        const jestFn = jest.fn();
        const SAMPLE_EVENT = 'someevent';
        const unsub = subscribe(videoElement, SAMPLE_EVENT, jestFn);

        videoElement.dispatchEvent(new Event(SAMPLE_EVENT));
        expect(jestFn).toBeCalled();

        const spyOnRemoveEventListener = jest.spyOn(videoElement, 'removeEventListener');
        unsub();

        // removeEventListener(event, listener); calls[0][i] is the ith argument
        expect(spyOnRemoveEventListener.mock.calls[0][0]).toEqual(SAMPLE_EVENT);
        expect(spyOnRemoveEventListener.mock.calls[0][1]).toEqual(jestFn);
    });
});
