import { BufferRange } from './mediasink';

export const UNKNOWN = -1;
export const MIN_PLAYABLE_BUFFER = 0.1;

/**
 * @returns {number} Number of frames decoded for video element
 * Returns -1 if decoded count unavailable.
 */
export function decodedFrames(videoElement: HTMLVideoElement): number {
    if (typeof videoElement.webkitDecodedFrameCount === 'number') {
        return videoElement.webkitDecodedFrameCount;
    } else if (typeof videoElement.getVideoPlaybackQuality === 'function') {
        return videoElement.getVideoPlaybackQuality().totalVideoFrames;
    } else if (typeof videoElement.mozDecodedFrames === 'number') {
        return videoElement.mozDecodedFrames;
    } else {
        return UNKNOWN;
    }
}

/**
 * Frames dropped in this stream. Reset in 'reset'
 * @return {number} Number of dropped frames
 */
export function droppedFrames(videoElement: HTMLVideoElement): number {
    if (typeof videoElement.webkitDroppedFrameCount === 'number') {
        return videoElement.webkitDroppedFrameCount;
    } else if (typeof videoElement.getVideoPlaybackQuality === 'function') {
        return videoElement.getVideoPlaybackQuality().droppedVideoFrames;
    } else {
        return UNKNOWN;
    }
}

/**
 * Returns buffered range containing the playhead
 * @return {Number} buffered.start - start of current buffer
 * @return {Number} buffered.end - end of current buffer
 */
export function bufferedRange(videoElement: HTMLVideoElement, minBuffer: number): BufferRange {
    const { buffered, currentTime } = videoElement;

    // Find buffered region containing playhead.
    // We only allow one buffered region.
    for (let i = 0; i < buffered.length; i++) {
        let start = buffered.start(i);
        let end = buffered.end(i);

        // keep looping until we find range containing playhead
        if (end <= currentTime) {
            continue;
        }

        // No region contains playhead.
        // Consider MIN_PLAYABLE_BUFFER before start as part of this region
        if (start - minBuffer > currentTime) {
            break;
        }

        // Include ranges past the end if there's a small gap
        for (let e = i + 1; e < buffered.length; e++) {
            if (buffered.start(e) - end > minBuffer) {
                break;
            }
            end = buffered.end(e);
        }

        // Include ranges before the start if there's a small gap
        for (let s = i - 1; s >= 0; s--) {
            if (start - buffered.end(s) > minBuffer) {
                break;
            }
            start = buffered.start(s);
        }

        return { start: Math.min(start, currentTime), end };
    }

    return { start: currentTime, end: currentTime };
}

export function willBuffer(videoElement: HTMLVideoElement, minBuffer: number) {
    const { end } = bufferedRange(videoElement, minBuffer);
    const playhead = videoElement.currentTime;

    const remaining = end - playhead;
    const hasEnded = videoElement.ended || (videoElement.duration - playhead < minBuffer);

    return !hasEnded && (remaining < minBuffer);
}

/**
 * Attempt to fix stalled playback. If stalled within a buffer, we try to "inch"
 * forward to get unstuck. This is most prevelent on Safari. If we're not in a
 * buffer, we'll attempt to seek forward to the next buffered region. This skips
 * any gaps that have formed or correct timestamp issues.
 */
export function getNextPlayablePosition(videoElement: HTMLVideoElement, minBuffer: number = MIN_PLAYABLE_BUFFER): number {
    const bufferDuration = bufferedRange(videoElement, minBuffer).end - videoElement.currentTime;
    const inBuffer = (bufferDuration > minBuffer);
    const buffered = videoElement.buffered;

    // Seek to the start of the next region if it exists.
    if (buffered.length > 1 || !inBuffer) {
        const playhead = videoElement.currentTime;
        for (let i = 0; i < buffered.length; i++) {
            const start = buffered.start(i);
            const end = buffered.end(i);

            if (playhead < start && end - start > minBuffer) {
                return start + minBuffer;
            }
        }
    }

    // Creep forward to try to get "unstuck"
    if (inBuffer) {
        return videoElement.currentTime + minBuffer;
    }

    return videoElement.currentTime;
}

export function subscribe(target: EventTarget, type: string, listener: EventListenerOrEventListenerObject): Function {
    const unsub = () => {
        target.removeEventListener(type, listener);
    };
    target.addEventListener(type, listener);
    return unsub;
}
