// tslint:disable:no-invalid-this
// tslint:disable:only-arrow-functions
// tslint:disable:no-any
// tslint:disable:no-invalid-this
import {
    createWorker,
    ErrorSource,
    ErrorType,
    MediaPlayer,
    MetadataEvent,
    PlayerEvent,
    PlayerState,
} from './index';
import { Quality } from './web-mediaplayer';

enum ReadyState {
    HAVE_NOTHING,
    HAVE_METADATA,
    HAVE_CURRENT_DATA,
    HAVE_FUTURE_DATA,
    HAVE_ENOUGH_DATA,
}

declare const WASM_WORKER_URL: string;
declare const ASMJS_WORKER_URL: string;
declare const WASM_BINARY_URL: string;

const useWasm = typeof window.WebAssembly === 'object' && typeof window.WebAssembly.instantiate === 'function';

const currentScript = document.currentScript as HTMLScriptElement;
const WORKER_URL = currentScript.src.replace(/[\w-\.]+\.js$/g, useWasm ? WASM_WORKER_URL : ASMJS_WORKER_URL);
const WASM_URL = currentScript.src.replace(/[\w-\.]+\.js$/g, WASM_BINARY_URL);

// tslint:disable-next-line:no-default-export
export = function (vjs: Videojs, techOptions: any) {
    if (typeof vjs === 'undefined' || typeof vjs.getTech !== 'function') {
        console.error('videojs not available, TwitchWhiteLabel tech not registered');
        return;
    }

    const WEB_WORKER = createWorker(WORKER_URL, WASM_URL);
    const Tech = vjs.getTech('Tech');

    const myTech = vjs.extend(Tech, {
        constructor(options, ready) {
            this._readyState = ReadyState.HAVE_NOTHING;

            this._mediaPlayer = new MediaPlayer({
                logLevel: techOptions.logLevel,
            }, WEB_WORKER);

            this._mediaPlayer.setAutoplay(options.autoplay === true);

            this._attachVideojsListeners();

            // Add captions to videojs text tracks by default.
            this._mediaPlayer.addEventListener(MetadataEvent.CAPTION, this._onCaptionEvent.bind(this));

            this._mediaPlayer.setAnalyticsSendEvents(true);

            this.featuresProgressEvents = true;
            this.featuresTimeupdateEvents = true;
            this.featuresPlaybackRate = true;
            this.featuresFullscreenResize = true;
            this.featuresVolumeControl = true;
            this.featuresMuteControl = true;
            this.featuresNativeTextTracks = false;

            Tech.call(this, options, ready);

            // Restore VTT implementation since we are going to use vttjs implementation directly
            window.vttjs.restore();

            this.triggerReady();
            // This setTimeout is necessary because other UI components
            // might not have attached their listeners yet
            setTimeout(function () {
                if (this.options_.loop) {
                    this._mediaPlayer.setLooping(true);
                }

                if (this.options_.muted) {
                    this._mediaPlayer.setMuted(true);
                }
            }.bind(this), 0);
        },
        dispose() {
            this._mediaPlayer.delete();
        },
        setPreload() {
            // No op
        },
        load() {
            // No op
        },
        readyState() {
            return this._readyState;
        },
        ended() {
            return this._mediaPlayer.getPlayerState() === PlayerState.ENDED;
        },
        seekable() {
            return this._mediaPlayer.getDuration() !== Infinity;
        },
        play() {
            this._mediaPlayer.play();
            this.trigger('play');
        },
        pause() {
            this._mediaPlayer.pause();
        },
        setCurrentTime(time) {
            this._mediaPlayer.seekTo(time);
            this.trigger('seeking');
        },
        controls() {
            // Native controls are not supported
            return false;
        },
        setControls() {
            // We don't allow native controls on video element.
            return false;
        },
        muted() {
            return this._mediaPlayer.isMuted();
        },
        setMuted(val: boolean) {
            this._mediaPlayer.setMuted(val);
        },
        volume() {
            return this._mediaPlayer.getVolume();
        },
        setVolume(val: number) {
            this._mediaPlayer.setVolume(val);
        },
        playbackRate() {
            return this._mediaPlayer.getPlaybackRate();
        },
        setPlaybackRate(rate: number) {
            this._mediaPlayer.setPlaybackRate(rate);
        },
        paused() {
            return this._mediaPlayer.isPaused();
        },
        duration() {
            return this._mediaPlayer.getDuration();
        },
        currentTime() {
            return this._mediaPlayer.getPosition();
        },
        createEl() {
            const videoElement = this._mediaPlayer.getHTMLVideoElement();
            videoElement.setAttribute('class', 'vjs-tech');

            ['preload', 'poster'].forEach(function (attr) {
                if (this.options_[attr]) {
                    videoElement.setAttribute(attr, this.options_[attr]);
                }
            }.bind(this));

            // Handle playsinline special
            if (this.options_.playsinline) {
                videoElement.setAttribute('webkit-playsinline', ''); // iOS < 10
                videoElement.setAttribute('playsinline', ''); // iOS >= 10
            }

            const videoDiv = document.createElement('div');
            videoDiv.appendChild(videoElement);
            return videoDiv;
        },
        src(srcUrl: string) {
            // remove caption track if any
            if (this._captionTrack) {
                this.textTracks().removeTrack(this._captionTrack);
                this._captionTrack = null;
            }

            if (srcUrl) {
                this._mediaPlayer.load(srcUrl);
            }
        },
        getQualities() {
            return this._mediaPlayer.getQualities();
        },
        setQuality(quality: Quality) {
            this._mediaPlayer.setQuality(quality);
        },
        getQuality() {
            return this._mediaPlayer.getQuality();
        },
        setAutoSwitchQuality(val: boolean) {
            this._mediaPlayer.setAutoSwitchQuality(val);
        },
        getAutoSwitchQuality() {
            return this._mediaPlayer.getAutoSwitchQuality();
        },
        getBufferDuration() {
            return this._mediaPlayer.getBufferDuration();
        },
        getDisplayHeight() {
            return this._mediaPlayer.getDisplayHeight();
        },
        getDisplayWidth() {
            return this._mediaPlayer.getDisplayWidth();
        },
        getDroppedFrames() {
            return this._mediaPlayer.getDroppedFrames();
        },
        getLiveLatency() {
            return this._mediaPlayer.getLiveLatency();
        },
        getVersion() {
            return this._mediaPlayer.getVersion();
        },
        getVideoBitRate() {
            return this._mediaPlayer.getVideoBitRate();
        },
        getVideoFrameRate() {
            return this._mediaPlayer.getVideoFrameRate();
        },
        getVideoHeight() {
            return this._mediaPlayer.getVideoHeight();
        },
        getVideoWidth() {
            return this._mediaPlayer.getVideoWidth();
        },
        isLiveLowLatency() {
            return this._mediaPlayer.isLiveLowLatency();
        },
        addEventListener(eventName, listener) {
            this._mediaPlayer.addEventListener(eventName, listener);
        },
        removeEventListener(eventName, listener) {
            this._mediaPlayer.removeEventListener(eventName, listener);
        },
        setLiveMaxLatency(latency) {
            this._mediaPlayer.setLiveMaxLatency(latency);
        },
        setAutoMaxVideoSize(width, height) {
            this._mediaPlayer.setAutoMaxVideoSize(width, height);
        },
        setAutoViewportSizeEnabled(enable) {
            this._mediaPlayer.setAutoViewportSizeEnabled(enable);
        },
        getMediaPlayerAPI() {
            return this._mediaPlayer;
        },
        supportsFullScreen() {
            return true;
        },
        enterFullScreen() {
            const videoElement = this._mediaPlayer.getHTMLVideoElement();
            const fullScreenFunction =
                videoElement.requestFullscreen ||
                videoElement.webkitRequestFullscreen ||
                videoElement.mozRequestFullScreen ||
                videoElement.msRequestFullscreen ||
                videoElement.webkitEnterFullscreen ||
                // tslint:disable-next-line:only-arrow-functions
                function () {
                    console.error('Fullscreen API is not available');
                };

            fullScreenFunction.call(videoElement);
        },
        exitFullScreen() {
            const exitFunction =
                document.exitFullScreen ||
                document.webkitExitFullscreen ||
                document.mozCancelFullScreen ||
                document.msExitFullscreen ||
                // tslint:disable-next-line:only-arrow-functions
                function () {
                    console.error('Exitscreen API is not available');
                };

            exitFunction.call(document);
        },
        // Special private functions
        _onCaptionEvent(caption) {
            // first time this event is fired
            if (!this._captionTrack) {
                this._captionTrack = this.addTextTrack('captions', caption.format);
                this._currentCue = null;
            }

            // remove exit cue
            if (this._currentCue) {
                this._captionTrack.removeCue(this._currentCue);
            }

            // Add the cue to track
            const videoElement = this._mediaPlayer.getHTMLVideoElement();
            const VTTCueClass: any = window.VTTCue || window.vttjs.VTTCue;
            if (VTTCueClass) {
                this._currentCue = new VTTCueClass(videoElement.currentTime, videoElement.currentTime + 2, caption.text);
                this._captionTrack.addCue(this._currentCue);
            } else {
                console.warn('No VTTCue implementation available, caption may not be available');
            }
        },
        _attachVideojsListeners() {
            this._mediaPlayer.addEventListener(PlayerState.READY, function () {
                this._readyState = ReadyState.HAVE_METADATA;
                this.trigger('loadedmetadata');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerState.IDLE, function () {
                this._readyState = ReadyState.HAVE_NOTHING;
                this.trigger('pause');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerState.PLAYING, function () {
                if (this._readyState <= ReadyState.HAVE_CURRENT_DATA) {
                    this._readyState = ReadyState.HAVE_FUTURE_DATA;
                }
                this.trigger('play'); // This is necessary in case we move to idle while switching streams
                this.trigger('playing');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerState.ENDED, function () {
                this._readyState = ReadyState.HAVE_NOTHING;
                this.trigger('ended');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerState.BUFFERING, function () {
                this._readyState = ReadyState.HAVE_CURRENT_DATA;
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerEvent.REBUFFERING, function () {
                this._readyState = ReadyState.HAVE_CURRENT_DATA;
                this.trigger('waiting');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerEvent.TIME_UPDATE, function () {
                this.trigger('timeupdate');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerEvent.VOLUME_CHANGED, function () {
                this.trigger('volumechange');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerEvent.MUTED_CHANGED, function () {
                this.trigger('volumechange');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerEvent.ERROR, function () {
                this.trigger('error');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerEvent.DURATION_CHANGED, function () {
                this.trigger('durationchange');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerEvent.SEEK_COMPLETED, function () {
                this.trigger('seeked');
            }.bind(this));

            this._mediaPlayer.addEventListener(PlayerEvent.PLAYBACK_RATE_CHANGED, function () {
                this.trigger('ratechange');
            }.bind(this));
        },
        techName: 'TwitchWhiteLabel',
        // Special exposure to events, error types and sources
        PlayerEvent,
        MetadataEvent,
        PlayerState,
        ErrorType,
        ErrorSource,
    });

    myTech.supportsFullScreen = function () {
        return true;
    };

    myTech.isSupported = function () {
        const appVersion = (navigator.appVersion || '').toLowerCase();

        return appVersion.indexOf('rv:11') === -1; // TODO remove when we have a solution for IE-11
    };

    myTech.canPlayType = function (type) {
        if (typeof type === 'string' && type.length > 0) {
            if (type.indexOf('application/x-mpegURL') > -1) {
                return typeof MediaSource !== 'undefined'
                    && MediaSource.isTypeSupported('video/mp4; codecs="avc1.42E01E,mp4a.40.2"');
            }

            const videoElement = document.createElement('video');
            return videoElement.canPlayType(type) !== '';
        }
        return false;
    };

    myTech.canPlaySource = function () {
        return true;
    };

    vjs.registerTech('TwitchWhiteLabel', myTech);

    // Register other plugins for quality
    const MenuButton = vjs.getComponent('MenuButton');
    const MenuItem = vjs.getComponent('MenuItem');

    const QualityMenuItem = vjs.extend(MenuItem, {
        constructor(player, options) {
            const qualityObject = options.quality;

            MenuItem.call(this, player, {
                selectable: true,
                label: qualityObject.name,
            });

            this.quality = qualityObject;
        },
        handleClick(event) {
            MenuItem.prototype.handleClick.call(this, event);
            if (this.quality.group === 'auto') {
                this.player().setAutoSwitchQuality(true);
            } else {
                this.player().setQuality(this.quality);
            }
        },
    });

    const QualityMenuButton = vjs.extend(MenuButton, {
        constructor(player, options) {
            MenuButton.call(this, player, {});
            this.controlText('Quality');
        },
        createItems() {
            const player = this.player();
            const menu = this.menu;
            const qualityItems = [];
            // Add the auto menu item
            const autoItem = new QualityMenuItem(player, {
                quality: {
                    group: 'auto',
                    name: 'Auto',
                },
            });
            autoItem.selected(player.getAutoSwitchQuality());
            // Add click/tap handler to all auto items
            autoItem.on('click', this._clickHandler.bind(this, autoItem));
            autoItem.on('tap', this._clickHandler.bind(this, autoItem));

            qualityItems.push(autoItem);

            const qualities = player.getQualities();
            if (qualities && qualities.length > 0) {
                qualities.forEach(function (q) {
                    const newItem = new QualityMenuItem(player, {
                        quality: q,
                    });

                    // Add click/tap handler to all menu items
                    newItem.on('click', this._clickHandler.bind(this, newItem));
                    newItem.on('tap', this._clickHandler.bind(this, newItem));

                    if (!player.getAutoSwitchQuality()) {
                        const currentQuality = player.getQuality();
                        newItem.selected(currentQuality.group === q.group);
                    }

                    qualityItems.push(newItem);
                }.bind(this));
            }

            return qualityItems;
        },
        buildCSSClass() {
            return 'vjs-icon-hd vjs-icon-placeholder ' + MenuButton.prototype.buildCSSClass.call(this);
        },
        _clickHandler(itemClicked) {
            // this.items => all other items added to this
            this.items.forEach(function (item) {
                if (item !== itemClicked) {
                    item.selected(false);
                }
            });
        },
    });

    vjs.registerComponent('QualityMenuButton', QualityMenuButton);

    const registerPlugin = vjs.registerPlugin || vjs.plugin;
    // Register plugin for enabling the quality menu button
    registerPlugin('enableTwitchPlugins', function () {
        const player = this;
        const tech = this.tech(true);

        tech.addEventListener(PlayerState.READY, function () {
            const qualityMenuButton = player.controlBar.getChild('QualityMenuButton');
            if (qualityMenuButton) {
                qualityMenuButton.dispose();
                player.controlBar.removeChild(qualityMenuButton);
            }
            player.controlBar.addChild('QualityMenuButton');
        });
    });

    registerPlugin('addTwitchTechEventListener', function (event, listener) {
        const tech = this.tech(true);
        tech.addEventListener(event, listener);
    });

    registerPlugin('removeTwitchTechEventListener', function (event, listener) {
        const tech = this.tech(true);
        tech.removeEventListener(event, listener);
    });

    registerPlugin('getTwitchTech', function () {
        return {
            PlayerEvent,
            MetadataEvent,
            PlayerState,
            ErrorType,
            ErrorSource,
        };
    });

    registerPlugin('getMediaPlayerAPI', function () {
        const tech = this.tech(true);
        return tech.getMediaPlayerAPI();
    });

    registerPlugin('getQualities', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getQualities();
    });

    registerPlugin('setQuality', function (quality: Quality) {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        mediaPlayerAPI.setQuality(quality);
    });

    registerPlugin('getQuality', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getQuality();
    });

    registerPlugin('setAutoSwitchQuality', function (val: boolean) {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        mediaPlayerAPI.setAutoSwitchQuality(val);
    });

    registerPlugin('getAutoSwitchQuality', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getAutoSwitchQuality();
    });

    registerPlugin('getBufferDuration', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getBufferDuration();
    });

    registerPlugin('getDisplayHeight', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getDisplayHeight();
    });

    registerPlugin('getDisplayWidth', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getDisplayWidth();
    });

    registerPlugin('getDroppedFrames', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getDroppedFrames();
    });

    registerPlugin('getLiveLatency', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getLiveLatency();
    });

    registerPlugin('getVersion', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getVersion();
    });

    registerPlugin('getVideoBitRate', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getVideoBitRate();
    });

    registerPlugin('getVideoFrameRate', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getVideoFrameRate();
    });

    registerPlugin('getVideoHeight', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getVideoHeight();
    });

    registerPlugin('getVideoWidth', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.getVideoWidth();
    });

    registerPlugin('isLiveLowLatency', function () {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        return mediaPlayerAPI.isLiveLowLatency();
    });

    registerPlugin('setLiveMaxLatency', function (latency: number) {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        mediaPlayerAPI.setLiveMaxLatency(latency);
    });

    registerPlugin('setAutoMaxVideoSize', function (width: number, height: number) {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        mediaPlayerAPI.setAutoMaxVideoSize(width, height);
    });

    registerPlugin('setAutoViewportSizeEnabled', function (enable: boolean) {
        const tech = this.tech(true);
        const mediaPlayerAPI = tech.getMediaPlayerAPI();
        mediaPlayerAPI.setAutoViewportSizeEnabled(enable);
    });
};
