var path = require('path');
var webpack = require('webpack');
var merge = require('webpack-merge');
var GitPlugin = require('git-revision-webpack-plugin');
var UglifyJsPlugin = require('uglifyjs-webpack-plugin');
var package = require('../../../package.json');
var nodeExternals = require('webpack-node-externals');

var PROD = (process.env.NODE_ENV === 'production');
var DEV_SERVER = process.argv.find(v => v.includes('webpack-dev-server'));
var FILE_EXT = (PROD ?  'min.js' : 'js');
var WASM_FILE_EXT = (PROD ?  'min.wasm' : 'wasm');
var GIT_COMMIT = process.env.GIT_COMMIT || new GitPlugin().commithash();
var PROJECT_DIR = path.resolve(__dirname, '../../../');

// Common configuration for all targets
var common = {
    mode: PROD ? 'production' : 'development',
    devtool: PROD ? false : 'inline-source-map',
    output: {
        path: path.resolve(PROJECT_DIR, 'artifacts')
    },
    resolve: {
        alias: {
            'settings': path.resolve(__dirname, '../settings'), // settings
        },
        extensions: ['.ts', '.js']
    },
    plugins: [
        new webpack.DefinePlugin({
            ASMJS_WORKER_URL: JSON.stringify(`worker.${FILE_EXT}`),
            WASM_WORKER_URL: JSON.stringify(`wasmworker.${FILE_EXT}`),
            WASM_BINARY_URL: JSON.stringify(`wasmworker.${WASM_FILE_EXT}`),
            VERSION: JSON.stringify(`${package.version}-${GIT_COMMIT.substring(0, 8)}`),
        }),
    ],
    node: {
        setImmediate: false,
        process: false,
        os: 'empty',
    },
    module: {
        rules: [
          {
            test: /\.ts$/,
            loader: 'ts-loader',
            exclude: /node_modules/,
            options: {
                configFile: path.resolve(__dirname, 'tsconfig.json')
            }
          }
        ]
    },
};

var workercommon = merge(common, {
    entry: [
        path.resolve(__dirname, 'workerpolyfill.ts'),
        path.resolve(__dirname, 'workermain.ts')
    ],
});

var wasmworker = merge(workercommon, {
    output: {
        filename: `wasmworker.${FILE_EXT}`,
    },
    resolve: {
        alias: {
            'emscriptenmodule': path.resolve(PROJECT_DIR, `build_js/platforms/web/wasmworker.${FILE_EXT}`), // settings
        },
    },
});

var asmjsworker = merge(workercommon, {
    output: {
        filename: `worker.${FILE_EXT}`,
    },
    optimization: {
        minimizer: [new UglifyJsPlugin({
            uglifyOptions: {
                compress: {
                    reduce_vars: false,
                },
            },
        })],
    },
    resolve: {
        alias: {
            'emscriptenmodule': path.resolve(PROJECT_DIR, `build_js/platforms/web/worker.${FILE_EXT}`), // settings
        }
    },
});

// Mediaplayer client
var mediaplayer = merge(common, {
    entry: path.resolve(__dirname, 'index.ts'),
    output: {
        libraryTarget: 'var',
        library: 'MediaPlayer',
        filename: `mediaplayer.${FILE_EXT}`
    }
});

var mediaplayerLib = merge(common, {
    entry: path.resolve(__dirname, 'index.ts'),
    output: {
        libraryTarget: 'commonjs2',
        library: 'MediaPlayerLib',
        filename: `mediaplayerlib.js`
    },
    target: 'node',
    externals: [nodeExternals()]
});

// Sample player with bundled mediaplayer client
var samplemediaplayer = merge(common, {
    entry: {
        samplemediaplayer: path.resolve(PROJECT_DIR, 'examples/web/samplemediaplayer.js'),
        testplayer: path.resolve(PROJECT_DIR, 'test/integration/app/web/testplayer.js'),
        'videojs-bundle': path.resolve(PROJECT_DIR, 'examples/web/videojs-bundle.js'),
    },
    output: {
        filename: DEV_SERVER ? '[name].js' : `[name].${FILE_EXT}`
    },
    devServer: {
        disableHostCheck: true,
        compress: true,
        contentBase: [path.resolve(PROJECT_DIR, 'examples/web'), path.resolve(PROJECT_DIR, 'test/integration/app/web'), path.resolve(PROJECT_DIR, 'artifacts')],
        before: app => {
            app.get(`/wasmworker.${WASM_FILE_EXT}`, (req, res, next) => {
                res.setHeader('content-type', 'application/wasm');
                next();
            });
        },
    },
});

// ChromecastMediaPlayer
var chromecast = merge(common, {
    entry: path.resolve(__dirname, 'chromecast-mediaplayer.js'),
    output: {
        libraryTarget: 'var',
        library: 'TwitchChromecastMediaPlayer',
        filename: `chromecast-mediaplayer.${FILE_EXT}`
    },
});

// Videojs
var videojs = merge(common, {
    entry: path.resolve(__dirname, 'videojs-hls-tech.ts'),
    output: {
        libraryTarget: 'var',
        library: 'registerTwitchTechForVideoJs',
        filename: `videojs-hls-tech.${FILE_EXT}`
    }
});

if (DEV_SERVER) {
    module.exports = [samplemediaplayer, wasmworker, videojs];
} else {
    module.exports = [mediaplayer, wasmworker, asmjsworker, chromecast, samplemediaplayer, videojs, mediaplayerLib];
}
