// tslint:disable-next-line:no-require-imports no-var-requires
const createEmscriptenModule = require('emscriptenmodule').default;
import { PlayerEvent } from './event/player';
import { sendFetchRequest } from './http';
import { MediaPlayerProxy, sendMessageToClient } from './mediaplayer-proxy';
import { scheduleTask } from './scheduler';
import { EmscriptenModule, MessageHandler } from './web-mediaplayer';

declare const self: DedicatedWorkerGlobalScope;
declare const messageHandler: MessageHandler | undefined;

interface WorkerConfig {
    wasmBinaryUrl: string;
}

interface WorkerMessage {
    id: number;
    funcName: string;
    // tslint:disable-next-line:no-any
    args?: any[];
}

// Allow users to overwrite the global 'onmessage' and 'postMessage'
// my defining a 'messageHandler' to use instead
setup(typeof messageHandler === 'undefined' ? self : messageHandler);

// Setup emscripten with initial worker message
function setup(port: MessageHander) {
    port.onmessage = (event) => {
        const dispatcher = new MessageDispatcher(port);
        port.onmessage = (evt) => dispatcher.dispatch(evt);
        initEmscripten(dispatcher, event.data);
    };
}

/**
 * Initialize emscripten with our javascript library functions.
 * Some of these depend on the initialized emscripten "Module"
 * object, so we'll create a circular reference to it here.
 */
function initEmscripten(dispatcher: MessageDispatcher, config: WorkerConfig) {
    const emModule: EmscriptenModule = createEmscriptenModule({
        scheduleTask,
        locateFile: () => config.wasmBinaryUrl,
        sendFetchRequest: (requestProxy, url, opts, timeout) =>
            sendFetchRequest(emModule, requestProxy, url, opts, timeout),
        onRuntimeInitialized: () => {
            // If we're synchronously initialized (asmjs),
            // wait for module initialization to complete
            if (emModule) {
                dispatcher.ready(emModule);
            } else {
                setTimeout(() => dispatcher.ready(emModule), 0);
            }
        },
    });
}

/**
 * MessageDispatcher forwards worker messages to the appropriate MediaPlayer
 * instance. Also handles the creation and delation of MediaPlayer instances
 */
class MessageDispatcher {
    private readonly activePlayers: Record<number, MediaPlayerProxy>;
    private readonly port: MessageHandler;
    private eventQueue: MessageEvent[] | null;
    private module: EmscriptenModule | null;

    constructor(port: MessageHandler) {
        this.activePlayers = Object.create(null);
        this.port = port;
        this.eventQueue = [];
        this.module = null;
    }

    /**
     * dispatch an event to the appropriate MediaPlayer instance.
     * @param  {Event} evt event object from the worker 'onmessage'
     */
    dispatch(evt: MessageEvent) {
        // Queue events if we're not ready
        if (this.module === null) {
            this.eventQueue.push(evt);
            return;
        }

        const { id, funcName, args }: WorkerMessage = evt.data;

        try {
            // 'create' is the only message that doesn't use
            // a current instance, so need to handle explicitly
            if (funcName === 'create') {
                const [config] = args;
                this.activePlayers[id] = new MediaPlayerProxy(this.port, id, (proxy) => {
                    return new this.module.WebMediaPlayer(proxy, config, config.browserContext);
                });
            } else if (funcName === 'runTests') {
                this.module.ccall('runTests', 'void', ['number'], [this.activePlayers[id].getPointer()]);
            } else {
                this.activePlayers[id].onClientMessage(funcName, args);

                // Need to do extra cleanup if we're deleting
                if (funcName === 'delete') {
                    delete this.activePlayers[id];
                }
            }
        } catch (err) {
            console.warn(err);
            sendMessageToClient(this.port, id, PlayerEvent.WORKER_ERROR, {
                message: err.message,
            });
        }
    }

    /**
     * mark that Emscripten is "ready".
     * @param  {Object} module The emscripten module object
     */
    ready(module: EmscriptenModule) {
        this.module = module;
        this.eventQueue.forEach(this.dispatch, this);
        this.eventQueue = null;
    }
}
