import { EventEmitter } from 'events';
import { Queue } from './queue';

type MessageHandler = (message: MessageEvent) => void;

export class WorkerShim {
    private workerPort: MessageHander;
    private emitter: EventEmitter;
    // tslint:disable-next-line:no-any
    private messageQueue: Queue<any> | null;
    private terminated: boolean;

    constructor(scriptFile: string, env: Window) {
        this.workerPort = {
            postMessage: this.postMessageFromWorker.bind(this),
            // tslint:disable-next-line:no-empty
            onmessage: () => {},
        };
        this.emitter = new EventEmitter();
        this.messageQueue = new Queue();
        this.terminated = false;
        this.loadScript(scriptFile, (script) => this.applyWorkerEnv(script, env));
    }

    // tslint:disable-next-line:no-any
    postMessage(message: any) {
        if (this.messageQueue) {
            this.messageQueue.push(message);
            return;
        }

        this.postMessageToWorker(message);
    }

    terminate() {
        this.terminated = false;
    }

    // tslint:disable-next-line:no-any
    addEventListener(name: string, fn: (...args: any[]) => void) {
        this.emitter.on(name, fn);
    }

    // tslint:disable-next-line:no-any
    removeEventListener(name: string, fn: (...args: any[]) => void) {
        this.emitter.off(name, fn);
    }

    private loadScript(url: string, onScript: (script: string) => void) {
        const xhr = new XMLHttpRequest();
        xhr.open('GET', url);
        xhr.addEventListener('load', () => {
            if (xhr.status >= 200 && xhr.status < 400) {
                onScript(xhr.response);
            } else {
                this.emitter.emit('error', new Error(xhr.statusText));
            }
        });
        xhr.addEventListener('error', (event) => {
            this.emitter.emit('error', event);
        });
        xhr.send();
    }

    private applyWorkerEnv(scriptText: string, env: Window) {
        // Apply worker in isolated environment. 'workerPort'
        // will be used to send and receive messages from worker.
        try {
            Function('self', 'messageHandler', scriptText)(window, this.workerPort);
        } catch (e) {
            this.emitter.emit('error', e);
            return;
        }

        while (!this.messageQueue.empty()) {
            this.postMessageToWorker(this.messageQueue.pop());
        }

        this.messageQueue = null;
    }

    // tslint:disable-next-line:no-any
    private postMessageFromWorker(message: any) {
        setTimeout(() => {
            this.emitter.emit('message', { data: message } as MessageEvent);
        }, 0);
    }

    // tslint:disable-next-line:no-any
    private postMessageToWorker(message: any) {
        setTimeout(() => {
            this.workerPort.onmessage({ data: message } as MessageEvent);
        }, 0);
    }
}
