#include "mp4packager.hpp"

namespace twitch {

void Packager::addSample(const MediaSampleBuffer& sample)
{
    m_buffer.insert(m_buffer.end(), sample.buffer.begin(), sample.buffer.end());
}

const std::vector<uint8_t>& Packager::buffer() const
{
    return m_buffer;
}

void Packager::clear()
{
    m_buffer.clear();
}

AudioPackager::AudioPackager(const std::vector<uint8_t>& extradata)
    : m_track(m_mp4.addTrackAac(std::micro::den, extradata, std::vector<media::EncryptionInfo>()))
{
}

void AudioPackager::addSample(const MediaSampleBuffer& sample)
{
    m_track->Mp4Track::addSample(sample);
    if (m_track->getDurationTime() > MediaTime::zero()) {
        m_mp4.renderTo(m_buffer);
    }
}

VideoPackager::VideoPackager(const std::vector<uint8_t>& extradata)
    : m_track(m_mp4.addTrackAvc(std::micro::den, extradata, std::vector<media::EncryptionInfo>()))
{
}

void VideoPackager::addSample(const MediaSampleBuffer& sample)
{
    processSample(sample);
    render();
}

void VideoPackager::processSample(const MediaSampleBuffer& sample)
{
    m_track->Mp4Track::addSample(sample);
}

void VideoPackager::render()
{
    if (m_track->getDurationTime() > MediaTime::zero()) {
        m_mp4.renderTo(m_buffer);
    }
}

VideoGopsPackager::VideoGopsPackager(const std::vector<uint8_t>& extradata)
    : VideoPackager(extradata)
{
}

void VideoGopsPackager::addSample(const MediaSampleBuffer& sample)
{
    // When flushing Gops, once we receive a sync frame, we know
    // we've already processed a full GOP, so we can render it to the buffer.
    const double MaxFragmentDuration = 30000; // milliseconds
    MediaTime duration(m_track->getDuration(), m_track->getTimescale());
    if (sample.isSyncSample || duration.milliseconds().count() >= MaxFragmentDuration) {
        render();
    }
    processSample(sample);
}

} // namespace twitch
