#include "web-http.hpp"
#include <chrono>
#include <emscripten/bind.h>
#include <emscripten/val.h>

static const int RESPONSE_ERROR = -1;
static const int READ_ERROR = -2;
static const int RESPONSE_TIMEOUT = -3;
static const int READ_TIMEOUT = -4;

namespace twitch {

// Pass data from fetch request to C++ handlers
struct ReaderProxy {
    ReaderProxy(HttpResponse::ContentHandler onBuf, HttpClient::ErrorHandler onErr)
        : m_onBuffer(std::move(onBuf))
        , m_onError(std::move(onErr))
    {
    }

    void read(uintptr_t ptr, size_t len)
    {
        m_onBuffer(reinterpret_cast<const uint8_t*>(ptr), len, false);
    }

    void end()
    {
        m_onBuffer(nullptr, 0, true);
    }

    void error(bool timeout)
    {
        m_onError(timeout ? READ_TIMEOUT : READ_ERROR);
    }

private:
    HttpResponse::ContentHandler m_onBuffer;
    HttpClient::ErrorHandler m_onError;
};

class WebHttpResponse : public HttpResponse {
public:
    WebHttpResponse(const emscripten::val& resp)
        : HttpResponse(resp.call<int>("getStatus"))
        , m_response(resp)
    {
    }
    ~WebHttpResponse() override = default;

    std::string getHeader(const std::string& key) const override
    {
        return m_response.call<std::string>("getHeader", key);
    }
    void read(ContentHandler onBuffer, ErrorHandler onError) override
    {
        using millis = std::chrono::duration<int, std::milli>;
        int timeoutMs = std::chrono::duration_cast<millis>(getReadTimeout()).count();
        auto proxy = new ReaderProxy(std::move(onBuffer), std::move(onError));
        // embind 'call' doesn't like raw pointers. Use prebound function to avoid this
        m_response["readBody"](proxy, timeoutMs);
    }

private:
    emscripten::val m_response; // 'Response' javascript object
};

// Pass response from fetch request to C++ handlers
struct RequestProxy {
    RequestProxy(HttpClient::ResponseHandler onResp, HttpClient::ErrorHandler onErr)
        : m_onResponse(std::move(onResp))
        , m_onError(std::move(onErr))
    {
    }

    // return 'false' to abort the request
    void response(const emscripten::val& resp)
    {
        m_onResponse(std::make_shared<WebHttpResponse>(resp));
    }

    void error(bool timeout)
    {
        m_onError(timeout ? RESPONSE_TIMEOUT : RESPONSE_ERROR);
    }

private:
    HttpClient::ResponseHandler m_onResponse;
    HttpClient::ErrorHandler m_onError;
};

class WebHttpRequest : public HttpRequest {
public:
    WebHttpRequest(const std::string& url, HttpMethod method)
        : HttpRequest(url, method)
        , m_headers(emscripten::val::object())
        , m_cancelFn(emscripten::val::null())
        , m_cancelled(false)
    {
    }
    ~WebHttpRequest() override = default;

    void cancel() override
    {
        m_cancelled = true;
        if (!m_cancelFn.isNull()) {
            m_cancelFn();
        }
    }
    void setHeader(const std::string& key, const std::string& value) override
    {
        m_headers.set(key, value);
    }

    void setContent(const std::vector<uint8_t>& content) override
    {
        m_content = content;
    }

    void send(HttpClient::ResponseHandler onResp, HttpClient::ErrorHandler onErr)
    {
        if (m_cancelled) {
            return;
        }

        auto opts = emscripten::val::object();
        opts.set("method", getMethodString());
        if (m_requestMode.length() > 0) {
            opts.set("mode", m_requestMode);
        }
        opts.set("headers", m_headers);

        // if cookie header is set we can't really send it instead turn on credentials
        if (m_headers.hasOwnProperty("Cookie")) {
            opts.set("credentials", "include");
        }
        if (!m_content.empty()) {
            opts.set("body", emscripten::typed_memory_view(m_content.size(), m_content.data()));
        }

        using millis = std::chrono::duration<int, std::milli>;
        int timeoutMs = std::chrono::duration_cast<millis>(getTimeout()).count();

        auto proxy = new RequestProxy(std::move(onResp), std::move(onErr));
        m_cancelFn = emscripten::val::module_property("sendFetchRequest")(proxy, getUrl(), opts, timeoutMs);
    }

private:
    emscripten::val m_headers;
    std::vector<uint8_t> m_content;
    emscripten::val m_cancelFn;
    bool m_cancelled;
};

std::shared_ptr<HttpRequest> WebHttpClient::createRequest(const std::string& url, HttpMethod method)
{
    return std::shared_ptr<HttpRequest>(new WebHttpRequest(url, method));
}

void WebHttpClient::send(std::shared_ptr<HttpRequest> req, ResponseHandler onResp, ErrorHandler onErr)
{
    static_cast<WebHttpRequest&>(*req).send(std::move(onResp), std::move(onErr));
}

using namespace emscripten;
EMSCRIPTEN_BINDINGS(webhttp)
{
    class_<RequestProxy>("RequestProxy")
        .function("response", &RequestProxy::response)
        .function("error", &RequestProxy::error);

    class_<ReaderProxy>("ReaderProxy")
        .function("read", &ReaderProxy::read)
        .function("end", &ReaderProxy::end)
        .function("error", &ReaderProxy::error);
}

} //namespace twitch
