#include "web-mediaplayer.hpp"
#include "conversion.hpp"
#include "player/MediaRequest.hpp"
#include <emscripten/bind.h>
#include <emscripten/emscripten.h>
#include <limits>

namespace twitch {
WebMediaPlayer::WebMediaPlayer(const emscripten::val& proxy,
    const emscripten::val& config,
    const BrowserContext& browserContext)
    : MediaPlayer(*this, std::make_shared<WebPlatform>(proxy, config, browserContext))
    , m_proxy(proxy)
    , m_listener(nullptr)
{
    setSettings(conversion::valToJson(config["settings"]));
}

// Player callbacks
void WebMediaPlayer::onDurationChanged(MediaTime duration)
{
    // numeric_limits<double>::infinity becomes 'Infinity' in javascript
    double seconds = (duration == MediaTime::max()
            ? std::numeric_limits<double>::infinity()
            : duration.seconds());
    m_proxy.call<void>("onDurationChanged", seconds);

    if (m_listener) {
        m_listener->onDurationChanged(duration);
    }
}

void WebMediaPlayer::onError(const Error& error)
{
    m_proxy.call<void>(
        "onError",
        std::string(mediaResultString(error.result)),
        error.result.code,
        std::string(errorSourceString(error.source)),
        error.message);

    if (m_listener) {
        m_listener->onError(error);
    }
}

void WebMediaPlayer::onMetadata(const std::string& type, const std::vector<uint8_t>& data)
{
    if (MediaType::Text_Json.name == type) {
        std::string text(data.data(), data.data() + data.size());
        m_proxy.call<void>("onJSONMetadata", conversion::utf8ToUtf16(text));
    }

    if (MediaType(type).isText()) {
        std::string text(data.data(), data.data() + data.size());
        m_proxy.call<void>("onMetadata", type, conversion::utf8ToUtf16(text));
    } else {
        m_proxy.call<void>("onMetadata", type, emscripten::typed_memory_view(data.size(), data.data()));
    }


    if (m_listener) {
        m_listener->onMetadata(type, data);
    }
}

void WebMediaPlayer::onQualityChanged(const Quality& quality)
{
    m_proxy.call<void>("onQualityChanged", quality);

    if (m_listener) {
        m_listener->onQualityChanged(quality);
    }
}

void WebMediaPlayer::onRebuffering()
{
    m_proxy.call<void>("onRebuffering");

    if (m_listener) {
        m_listener->onRebuffering();
    }
}

void WebMediaPlayer::onRecoverableError(const Error& error)
{
    m_proxy.call<void>(
        "onRecoverableError",
        std::string(mediaResultString(error.result)),
        error.result.code,
        std::string(errorSourceString(error.source)),
        error.message);

    if (m_listener) {
        m_listener->onRecoverableError(error);
    }
}

void WebMediaPlayer::onSeekCompleted(MediaTime time)
{
    m_proxy.call<void>("onSeekCompleted");

    if (m_listener) {
        m_listener->onSeekCompleted(time);
    }
}

void WebMediaPlayer::onSessionData(const std::map<std::string, std::string>& properties)
{
    m_sessionData = properties;
    auto props = emscripten::val::object();
    for (const auto& prop : properties) {
        props.set(prop.first, prop.second);
    }
    m_proxy.call<void>("onSessionData", props);

    if (m_listener) {
        m_listener->onSessionData(properties);
    }
}

void WebMediaPlayer::onStateChanged(Player::State state)
{
    m_proxy.call<void>("onStateChanged", stateUpdate(state));

    if (state == Player::State::Ready) {
        const auto holdbackIter = m_sessionData.find("B");
        const auto countryIter = m_sessionData.find("USER-COUNTRY");
        const auto end = m_sessionData.end();
        // Consider in holdback group if key isn't present
        const bool isHoldback = (holdbackIter == end || holdbackIter->second == "true");
        const bool isKorea = (countryIter != end && countryIter->second == "KR");

        if (!isHoldback && isKorea) {
            // Remove source from auto
            const auto& qualities = getQualities();
            if (!qualities.empty()) {
                setAutoMaxBitrate(qualities[0].bitrate - 1);
            }
            m_proxy.call<void>("postMessage", std::string("2020"));
        }
    }

    if (m_listener) {
        m_listener->onStateChanged(state);
    }
}

void WebMediaPlayer::onAnalyticsEvent(const std::string& name, const std::string& properties)
{
    m_proxy.call<void>("onAnalyticsEvent", name, properties);

    if (m_listener) {
        m_listener->onAnalyticsEvent(name, properties);
    }
}

void WebMediaPlayer::onPositionChanged(MediaTime position)
{
    if (m_listener) {
        m_listener->onPositionChanged(position);
    }
}

void WebMediaPlayer::onRequestSent(const MediaSource::Request& request)
{
    MediaPlayer::onRequestSent(request);
    if (request.getType() == MediaRequest::Type::MasterPlaylist) {
        m_proxy.call<void>("onMasterPlaylistRequest");
    } else if (request.getType() == MediaRequest::Type::MediaPlaylist) {
        m_proxy.call<void>("onMediaPlaylistRequest");
    }
}

void WebMediaPlayer::onResponseReceived(const MediaSource::Request& request)
{
    MediaPlayer::onResponseReceived(request);
    if (request.getType() == MediaRequest::Type::MasterPlaylist) {
        m_proxy.call<void>("onMasterPlaylistReady");
    } else if (request.getType() == MediaRequest::Type::MediaPlaylist) {
        m_proxy.call<void>("onMediaPlaylistReady");
    }
}

void WebMediaPlayer::onResponseBytes(const MediaSource::Request& request, size_t bytes)
{
    MediaPlayer::onResponseBytes(request, bytes);
}

void WebMediaPlayer::onResponseEnd(const MediaSource::Request& request)
{
    MediaPlayer::onResponseEnd(request);
}

void WebMediaPlayer::onRequestError(const MediaSource::Request& request, int error)
{
    MediaPlayer::onRequestError(request, error);
}

// MediaSink callbacks
void WebMediaPlayer::onClientSinkDurationChanged(double time)
{
    getSink().onSinkDurationChanged(time);
}

void WebMediaPlayer::onClientSinkUpdate(const emscripten::val& update)
{
    getSink().onSinkUpdate(update);
}

void WebMediaPlayer::onClientSinkIdle()
{
    getSink().onSinkIdle();
}

void WebMediaPlayer::onClientSinkPlaying()
{
    getSink().onSinkPlaying();
}

void WebMediaPlayer::onClientSinkEnded()
{
    getSink().onSinkEnded();
}

void WebMediaPlayer::onClientSinkError(int value, int code, const std::string& message)
{
    getSink().onSinkError(value, code, message);
}

void WebMediaPlayer::onClientSinkCue(int id, bool enter)
{
    getSink().onSinkCue(id, enter);
}

void WebMediaPlayer::onMediaDecodingInfoResult(const MediaCapabilitiesInfo& info)
{
    (void)info;
}

void WebMediaPlayer::getDecodingInfo(const MediaConfig& config)
{
    m_proxy.call<void>("getDecodingInfo", config);
}

emscripten::val WebMediaPlayer::getStats() const
{
    auto segStats = emscripten::val::array();
    for (const auto& stat : getQualitySelector().getTransferHistory()) {
        auto obj = emscripten::val::object();
        obj.set("startTime", static_cast<double>(stat.initiated.milliseconds().count()));
        obj.set("downloadDuration", static_cast<int>(stat.getDuration().milliseconds().count()));
        obj.set("bytes", stat.bytes);
        obj.set("segmentDuration", static_cast<int>(stat.mediaDuration.milliseconds().count()));
        obj.set("firstByteLatency", static_cast<int>(stat.getRequestLatency().milliseconds().count()));
        segStats.call<void>("push", obj);
    }
    const auto& latencyStats = getLatencyStatistics();

    auto stats = emscripten::val::object();
    stats.set("videoBitRate", getSink().videoBitRate());
    stats.set("averageBitrate", getAverageBitrate());
    stats.set("bandwidthEstimate", getBandwidthEstimate());
    stats.set("networkProfile", segStats);
    stats.set("broadcasterLatency", getLiveLatency().seconds());
    // TODO deprecate transcoderLatency
    stats.set("transcoderLatency", latencyStats.getTranscoderLatency().seconds());
    return stats;
}

void WebMediaPlayer::setListener(Player::Listener* listener)
{
    m_listener = listener;
}

// Return values updated from this player state transition
emscripten::val WebMediaPlayer::stateUpdate(MediaPlayer::State playerState) const
{
    auto state = emscripten::val::object();
    state.set("playerState", stateToString(playerState));
    switch (playerState) {
    case MediaPlayer::State::Ready:
        state.set("quality", getQuality());
        state.set("qualities", getQualitiesArray());
        break;
    default:
        break;
    }
    return state;
}

emscripten::val WebMediaPlayer::getQualitiesArray() const
{
    auto qualities = emscripten::val::array();

    for (const auto& quality : getQualities()) {
        qualities.call<void>("push", quality);
    }

    return qualities;
}

WebMediaSink& WebMediaPlayer::getSink() const
{
    return static_cast<WebMediaSink&>(MediaPlayer::getSink());
}
}
