#include "web-mediasink.hpp"
#include "conversion.hpp"
#include "mp4packager.hpp"
#include "player/VideoStatistics.hpp"
#include <emscripten/bind.h>
#include <json11.hpp>

namespace twitch {

WebMediaSink::WebMediaSink(MediaSink::Listener& listener, const emscripten::val& proxy, bool flushGops)
    : m_metaTrackID(-1)
    , m_formatTrackID(-1)
    , m_statsTrackID(-1)
    , m_nextCueID(0)
    , m_currentTime(0.0)
    , m_playbackRate(1.0f)
    , m_listener(listener)
    , m_proxy(proxy)
    , m_mode(MSE) // TODO: pass in mode in here
    , m_flushGops(flushGops)
    , m_formatChanged(false)
    , m_blockFlush(false)
    , m_wasClear(false)
{
}

WebMediaSink::~WebMediaSink() {}

bool isPassthroughMediaType(const MediaType& mediaType)
{
    return mediaType.matches(MediaType::Audio_MP4) || mediaType.matches(MediaType::Video_MP4);
}

std::string modeToString(WebMediaSink::Mode mode)
{
    switch (mode) {
    case WebMediaSink::MSE:
        return "mse";
    case WebMediaSink::Passthrough:
        return "passthrough";
    case WebMediaSink::Chromecast:
        return "chromecast";
    }
}

void WebMediaSink::configure(int trackID, std::shared_ptr<MediaFormat> format)
{
    const auto& mediaType = format->getType();

    if (mediaType.isText()) {
        m_metaTrackID = trackID;
        return;
    }

    m_formats[trackID] = format;
    const bool isProtected = !format->getProtectionData().empty();

    // 'currentFormat' falsy on first call to configure after 'reset'.
    // Only need to block flush if we've played unprotected content
    // and are switching to DRM content
    if (format->getType().isVideo() && m_wasClear && isProtected) {
        m_blockFlush = true;
    }

    // If we're reconfiguring an existing track, flush any buffered
    // media. We render first incase we're in GoPs mode
    auto existingTrack = m_tracks.find(trackID);
    if (existingTrack != m_tracks.end()) {
        flushTrack(*existingTrack);
    }

    if (isPassthroughMediaType(mediaType)) {
        m_formatChanged = true;
        m_formatTrackID = trackID;
        m_statsTrackID = trackID;
        if (!m_blockFlush) {
            m_tracks[trackID].reset(new Packager()); // Just pass through the mp4
        }
    } else if (mediaType.matches(MediaType::Audio_AAC)) {
        m_formatChanged = true;
        m_formatTrackID = trackID;

        if (format->hasCodecData(MediaFormat::Audio_AAC_ESDS)) {
            m_tracks[trackID].reset(new AudioPackager(format->getCodecData(MediaFormat::Audio_AAC_ESDS)));
        }
    } else if (mediaType.matches(MediaType::Video_AVC)) {
        // Collect stats from the video track
        m_statsTrackID = trackID;

        if (format->hasCodecData(MediaFormat::Video_AVC_AVCC)) {
            if (m_flushGops) {
                m_tracks[trackID].reset(new VideoGopsPackager(format->getCodecData(MediaFormat::Video_AVC_AVCC)));
            } else {
                m_tracks[trackID].reset(new VideoPackager(format->getCodecData(MediaFormat::Video_AVC_AVCC)));
            }
        }
    }

    const std::string& path = format->getPath();

    const bool isPassthrough = (mediaType.matches(MediaType::Application_MPEG_URL)
        || (mediaType.matches(MediaType::Video_MP4) && path.find(".mp4") != std::string::npos));

    if (isPassthrough) {
        m_mode = (format->getName() == "remote") ? Chromecast : Passthrough;
    } else {
        m_mode = MSE;
    }

    m_proxy.call<void>("configure", trackID, mediaType.parameters, path, modeToString(m_mode), isProtected);
}

void WebMediaSink::endOfStream()
{
    for (const auto& track : m_tracks) {
        flushTrack(track);
    }
    m_proxy.call<void>("endOfStream");
}

void WebMediaSink::enqueue(int trackID, std::shared_ptr<MediaSampleBuffer> samplePtr)
{
    const auto& sample = *samplePtr;

    if (trackID == m_metaTrackID) {
        // guarantee minimum cue duration to make sure it isn't skipped
        double start = sample.presentationTime.seconds();
        double end = start + std::max(1.0, sample.duration.seconds());
        addCue(start, end, [this, samplePtr](bool enter) {
            this->handleCue(enter, *samplePtr);
        });
        return;
    }

    auto trackItr = m_tracks.find(trackID);
    if (trackItr != m_tracks.end()) {
        const auto& track = *trackItr;
        auto currentFormat = m_formats[trackID];
        if (sample.isDiscontinuity && sample.duration > MediaTime::zero() && currentFormat && isPassthroughMediaType(currentFormat->getType())) {
            flushTrack(track); // Flush existing media before applying new offset
            MediaTime offset = sample.presentationTime - sample.decodeTime;
            m_proxy.call<void>("setTimestampOffset", trackID, offset.seconds());
        }
        if (currentFormat->getType().isVideo()) {
            m_wasClear = currentFormat->getProtectionData().empty();
        }
        track.second->addSample(sample);
    }

    if (m_formatChanged && trackID == m_formatTrackID && sample.duration > MediaTime::zero()) {
        m_formatChanged = false;
        double start = sample.presentationTime.seconds();
        // Clamp end to current time to guarantee its fired. Add a small factor
        // to account for playhead movement since the last update
        double end = std::max(start + sample.duration.seconds(), m_currentTime + 0.5);
        auto currentFormat = m_formats[trackID];
        addCue(start, end, [this, currentFormat](bool enter) {
            if (enter) {
                m_listener.onSinkFormatChanged(*currentFormat);
            }
        });
    }

    if (trackID == m_statsTrackID) {
        m_stats.addSample(sample);
        // Send video stats on key frames
        if (sample.isSyncSample) {
            m_proxy.call<void>("sendStats");
        }
    }
}

void WebMediaSink::flush()
{
    for (const auto& track : m_tracks) {
        flushTrack(track, false);
    }
}

void WebMediaSink::play()
{
    m_proxy.call<void>("play");
}

void WebMediaSink::pause()
{
    m_proxy.call<void>("pause");
}

void WebMediaSink::reset()
{
    m_metaTrackID = -1;
    m_formatTrackID = -1;
    m_statsTrackID = -1;
    m_nextCueID = 0;
    m_blockFlush = false;
    m_wasClear = false;
    m_formatChanged = false;
    m_formats.clear();
    m_tracks.clear();
    m_metaSamples.clear();
    m_proxy.call<void>("reset");
    // Reset the playback rate after every stream
    setPlaybackRate(1.0f);
}

void WebMediaSink::remove(const TimeRange& range)
{
    // Remove in range [start, end)
    double start = range.start.seconds();
    double end = range.end().seconds();
    m_proxy.call<void>("remove", start, end);

    for (auto iter = m_metaSamples.begin(); iter != m_metaSamples.end();) {
        auto& sample = iter->second;
        if (start <= sample.start && sample.start < end) {
            iter = m_metaSamples.erase(iter);
        } else {
            iter++;
        }
    }
}

void WebMediaSink::seekTo(MediaTime time)
{
    m_proxy.call<void>("seekTo", time.seconds());
}

void WebMediaSink::setPlaybackRate(float rate)
{
    if (m_playbackRate != rate) {
        m_playbackRate = rate;
        m_proxy.call<void>("setPlaybackRate", rate);
    }
}

void WebMediaSink::setVolume(float volume)
{
    (void)volume; // We handle this in the client for now
}

double WebMediaSink::videoBitRate() const
{
    return m_stats.videoBitRate;
}

void WebMediaSink::onSinkDurationChanged(double time)
{
    m_listener.onSinkDurationChanged(MediaTime(time));
}

void WebMediaSink::onSinkUpdate(const emscripten::val& update)
{
    VideoStatistics videoStats;
    videoStats.setBitRate(videoBitRate());
    videoStats.setFrameRate(int(update["framerate"].as<double>() + 0.5));
    videoStats.setDroppedFrames(update["droppedFrames"].as<int>());
    videoStats.setDecodedFrames(update["decodedFrames"].as<int>());
    videoStats.setRenderedFrames(videoStats.getDecodedFrames() - videoStats.getDroppedFrames());
    m_listener.onSinkVideoStatistics(videoStats);

    m_currentTime = update["currentTime"].as<double>();
    m_listener.onSinkTimeUpdate(MediaTime(m_currentTime));
}

void WebMediaSink::onSinkEnded()
{
    m_listener.onSinkStateChanged(MediaSink::State::Ended);
}

void WebMediaSink::onSinkIdle()
{
    if (m_blockFlush) {
        resumeFlush();
    } else {
        m_listener.onSinkStateChanged(MediaSink::State::Idle);
    }
}

void WebMediaSink::onSinkPlaying()
{
    m_listener.onSinkStateChanged(MediaSink::State::Playing);
}

void WebMediaSink::onSinkError(int value, int code, const std::string& message)
{
    // MediaErrors https://developer.mozilla.org/en-US/docs/Web/API/MediaError
    const int MEDIA_ERR_ABORTED = 1;
    const int MEDIA_ERR_NETWORK = 2;
    const int MEDIA_ERR_DECODE = 3;
    const int MEDIA_ERR_SRC_NOT_SUPPORTED = 4;
    const int MEDIA_ERR_SRC_BUFFERING_TIMEOUT = 101;

    MediaResult result = MediaResult::Error;
    ErrorSource source = ErrorSource::Unspecified;
    switch (value) {
    case MEDIA_ERR_ABORTED:
        result = MediaResult::ErrorInvalidState;
        break;
    case MEDIA_ERR_NETWORK:
        result = MediaResult::ErrorNetwork;
        break;
    case MEDIA_ERR_DECODE:
        result = MediaResult::ErrorInvalidData;
        source = ErrorSource::Decode;
        break;
    case MEDIA_ERR_SRC_NOT_SUPPORTED:
        result = MediaResult::ErrorNotSupported;
        break;
    case MEDIA_ERR_SRC_BUFFERING_TIMEOUT:
        result = MediaResult::ErrorTimeout;
        source = ErrorSource::Render;
    }
    result.code = code; // set platform specific code
    m_listener.onSinkError(Error(source, result, message));
}

void WebMediaSink::onSinkCue(int id, bool enter)
{
    auto it = m_metaSamples.find(id);
    if (it != m_metaSamples.end()) {
        it->second.fn(enter);
    }
}

void WebMediaSink::handleCue(bool enter, const MediaSampleBuffer& sample)
{
    if (sample.type != media::fourcc("json")) {
        if (enter) {
            m_listener.onSinkMetadataSample(sample);
        }
        return;
    }

    std::string err;
    std::string content(sample.buffer.begin(), sample.buffer.end());
    auto parsed = json11::Json::parse(content, err);

    if (!err.empty()) {
        return;
    }

    auto& object = parsed.object_items();
    const auto objEnd = object.end();

    // Stop blocking flush if we received end of stitched ad
    auto classEntry = object.find("CLASS");
    if (m_blockFlush && !enter && classEntry != objEnd && classEntry->second == "twitch-stitched-ad") {
        resumeFlush();
    }

    bool isLegacy = (object.find("ID3") != objEnd || object.find("caption") != objEnd);

    if (isLegacy) {
        if (enter) {
            m_listener.onSinkMetadataSample(sample);
        }
    } else {
        // We'll just emit METADATA_CUE_(ENTER|EXIT) events directly from here
        // TODO: make these not web-only and replace all other timed metadata
        // with these events.
        const char* type = enter ? "meta_enter" : "meta_exit";
        std::string metaJson;
        json11::Json(json11::Json::object{ { type, parsed } }).dump(metaJson);
        m_proxy.call<void>("onJSONMetadata", conversion::utf8ToUtf16(metaJson));
        if (enter) {
            m_listener.onSinkMetadataSample(sample);
        }
    }
}

void WebMediaSink::flushTrack(const TrackMap::value_type& track, bool render)
{
    Packager& packager = *track.second;

    if (render) {
        packager.render();
    }

    const auto& buffer = packager.buffer();

    if (!m_blockFlush && !buffer.empty()) {
        m_proxy.call<void>("enqueue", track.first, emscripten::typed_memory_view(buffer.size(), buffer.data()));
        packager.clear();
    }
}

void WebMediaSink::resumeFlush()
{
    m_blockFlush = false;
    m_proxy.call<void>("reinit");
    for (const auto& track : m_tracks) {
        flushTrack(track, false);
    }
    play();
}

void WebMediaSink::addCue(double start, double end, OnCueFn onCue)
{
    int id = m_nextCueID;
    m_nextCueID++;
    m_metaSamples[id] = { start, end, std::move(onCue) };
    m_proxy.call<void>("addCue", id, start, end);
}

void WebMediaSink::StatsCollector::addSample(const MediaSampleBuffer& sample)
{
    // Update once per GoP
    if (sample.isSyncSample && m_segmentDuration != MediaTime::zero()) {
        double segDur = m_segmentDuration.seconds();
        this->videoBitRate = (m_segmentBytes * 8) / segDur;
        m_segmentBytes = 0;
        m_segmentDuration = MediaTime::zero();
    }

    m_segmentBytes += static_cast<int>(sample.buffer.size());
    m_segmentDuration += sample.duration;
}

} //namespace twitch
