#pragma once
#include "playercore/MediaSink.hpp"
#include <emscripten/val.h>
#include <map>
#include <memory>

namespace twitch {

class Packager;

class WebMediaSink : public MediaSink {
public:
    WebMediaSink(MediaSink::Listener& listener, const emscripten::val& proxy, bool flushGops);
    virtual ~WebMediaSink();

    void configure(int trackID, std::shared_ptr<MediaFormat> format) override;
    void endOfStream() override;
    void enqueue(int trackID, std::shared_ptr<MediaSampleBuffer> sample) override;
    void flush() override;
    void play() override;
    void prepare() override {}
    void pause() override;
    void remove(const TimeRange& range) override;
    void reset() override;
    void seekTo(MediaTime time) override;
    void setPlaybackRate(float rate) override;
    void setSurface(void* surface) override { (void)surface; };
    void setVolume(float volume) override;
    double videoBitRate() const;

    // Forward MediaSink events to listeners
    void onSinkDurationChanged(double time);
    void onSinkUpdate(const emscripten::val& update);
    void onSinkEnded();
    void onSinkIdle();
    void onSinkPlaying();
    void onSinkError(int value, int code, const std::string& message);
    void onSinkCue(int id, bool enter);

    enum Mode {
        MSE,
        Passthrough,
        Chromecast
    };

private:
    using OnCueFn = std::function<void(bool)>;
    using TrackMap = std::map<int, std::unique_ptr<Packager>>;
    struct Cue {
        double start;
        double end;
        OnCueFn fn;
    };

    void handleCue(bool enter, const MediaSampleBuffer& sample);
    void flushTrack(const TrackMap::value_type& track, bool render = true);
    void resumeFlush();
    void addCue(double start, double end, OnCueFn onCue);

    // Calculate bitrate and fps from stream of samples
    struct StatsCollector {
        double videoBitRate = 0.0;
        void addSample(const MediaSampleBuffer&);

    private:
        int m_segmentBytes = 0;
        MediaTime m_segmentDuration = MediaTime::zero();
    };

    StatsCollector m_stats;
    int m_metaTrackID;
    int m_formatTrackID;
    int m_statsTrackID;
    int m_nextCueID;
    TrackMap m_tracks;
    double m_currentTime;
    float m_playbackRate;
    std::map<int, std::shared_ptr<MediaFormat>> m_formats;
    std::map<int, Cue> m_metaSamples;
    MediaSink::Listener& m_listener;
    emscripten::val m_proxy;
    Mode m_mode;
    bool m_flushGops;
    bool m_formatChanged;
    bool m_blockFlush;
    bool m_wasClear;
};

} //namespace twitch
