#include "web-platform.hpp"
#include "conversion.hpp"
#include "debug/trace.hpp"
#include "debug/TraceLog.hpp"
#include "msereaderfactory.hpp"
#include "player/PassthroughSource.hpp"
#include "player/Settings.hpp"
#include "util/Uuid.hpp"
#include "web-http.hpp"
#include "web-log.hpp"
#include "web-mediasink.hpp"
#include "web-scheduler.hpp"

namespace twitch {

Capabilities initCapabilities(const BrowserContext& browserContext, const emscripten::val& config)
{
    Settings settings(conversion::valToJson(config["settings"]));

    Capabilities capabilities;
    capabilities.avcFormat = AVCFormatType::AVCC;
    capabilities.supportsSingleFrameFragments = (!browserContext.isChromecast
        && ((browserContext.isChrome && browserContext.major >= 50)
            || (browserContext.isFirefox && browserContext.major >= 45)
            || (browserContext.isSafari && browserContext.major >= 11)));
    capabilities.supportsLowLatencyABR = settings.get("abr", "enableLowLatency", browserContext.isChrome || browserContext.isFirefox);
    capabilities.supportsSpeedUp = !browserContext.isChromecast;
    capabilities.supportsHttpResponseTiming = browserContext.isChrome || browserContext.isChromecast;
    capabilities.supportsBufferOverwrite = browserContext.isChrome;
    return capabilities;
}

std::map<std::string, std::string> initAnalyticProperties(const BrowserContext& browserContext)
{
    std::map<std::string, std::string> properties;
    properties["user_agent"] = browserContext.userAgent;
    properties["browser_family"] = browserContext.family;
    properties["browser_version"] = std::to_string(browserContext.major) + "." + std::to_string(browserContext.minor);
    properties["browser"] = browserContext.name;
    properties["os_name"] = browserContext.osName;
    properties["os_version"] = browserContext.osVersion;
    properties["url"] = browserContext.url;
    properties["host"] = browserContext.host;
    properties["domain"] = browserContext.domain;
    return properties;
}

WebPlatform::WebPlatform(const emscripten::val& proxy, const emscripten::val& config, const BrowserContext& browserContext)
    : m_proxy(proxy)
    , m_webLog(std::make_shared<WebLog>(config["logLevel"].as<std::string>()))
    , m_webHTTP(std::make_shared<WebHttpClient>())
    , m_readerFactory(std::make_shared<MSEMediaReaderFactory>())
    , m_supported({ MediaType::Video_AVC, MediaType::Audio_AAC, MediaType::Video_MP4 })
    , m_mseSupported(config["mseSupported"].as<bool>())
    , m_capabilities(initCapabilities(browserContext, config))
    , m_analyticsProperties(initAnalyticProperties(browserContext))
{
    // Try native HLS playback if no MSE
    if (!m_mseSupported) {
        m_supported.insert(MediaType::Application_MPEG_URL);
    }

    // Add protection systems based on browser
    std::string keySystem = config["keySystem"].as<std::string>();
    if (!keySystem.empty()) {
        m_protectionSystems.insert(Uuid::fromString(keySystem).toBytes());
    }

    std::vector<std::string> codecs = emscripten::vecFromJSArray<std::string>(config["codecs"]);
    for (const auto& codec : codecs) {
        if (codec == "vp09") {
            m_supported.insert(MediaType::Video_VP9);
        }
    }
}

std::shared_ptr<Scheduler> WebPlatform::createScheduler(const std::string& name)
{
    (void)name;
    return std::make_shared<WebScheduler>();
}

std::unique_ptr<MediaSource> WebPlatform::createSource(const std::string& path, const MediaType& mediaType,
    MediaSource::Listener& listener,
    std::shared_ptr<Scheduler> scheduler)
{
    (void)scheduler;
    if (MediaType::Video_MP4.matches(mediaType) || (!m_mseSupported && MediaType::Application_MPEG_URL.matches(mediaType))) {
        return std::unique_ptr<MediaSource>(new PassthroughSource(listener, mediaType, path));
    }

    return nullptr;
}

std::shared_ptr<HttpClient> WebPlatform::createAsyncHttpClient(std::shared_ptr<Scheduler>)
{
    // HttpClient already async on web
    return getHttpClient();
}

std::unique_ptr<MediaSink> WebPlatform::createSink(MediaSink::Listener& listener, std::shared_ptr<Scheduler>)
{
    // TODO: Pass 'mode' into WebMediaSink here
    return std::unique_ptr<WebMediaSink>(new WebMediaSink(listener, m_proxy, !m_capabilities.supportsSingleFrameFragments));
}

std::shared_ptr<Log> WebPlatform::getLog() const
{
    return m_webLog;
}

std::shared_ptr<HttpClient> WebPlatform::getHttpClient() const
{
    return m_webHTTP;
}

std::shared_ptr<MediaReaderFactory> WebPlatform::getMediaReaderFactory() const
{
    return m_readerFactory;
}

const std::string& WebPlatform::getName() const
{
    static const std::string name = "web";
    return name;
}

const Capabilities& WebPlatform::getCapabilities() const
{
    return m_capabilities;
}

const std::set<MediaType>& WebPlatform::getSupportedMediaTypes() const
{
    return m_supported;
}

const std::set<std::vector<uint8_t>>& WebPlatform::getSupportedProtectionSystems() const
{
    return m_protectionSystems;
}

const std::map<std::string, std::string>& WebPlatform::getAnalyticsProperties()
{
    return m_analyticsProperties;
};

VideoDecoderCapabilities WebPlatform::getVideoDecoderCapabilities(const MediaType& mediaType)
{
    (void)mediaType;
    return VideoDecoderCapabilities();
}

} //namespace twitch
