#include "web-scheduler.hpp"
#include <algorithm>
#include <chrono>
#include <emscripten/bind.h>
#include <emscripten/val.h>

namespace twitch {

class WebTask : public Cancellable {
public:
    WebTask(Scheduler::Action action)
        : m_action(std::move(action))
        , m_cancel(emscripten::val::undefined())
    {
    }

    void cancel() override
    {
        m_cancel();
    }

    void call()
    {
        m_action();
    }

    // Set the 'cancel' function. Used internally
    // by the WebScheduler implementation
    void setCancel(const emscripten::val& cancel)
    {
        m_cancel = cancel;
    }

private:
    Scheduler::Action m_action;
    emscripten::val m_cancel;
};

class NoopCancel : public Cancellable {
public:
    virtual ~NoopCancel() = default;
    void cancel() override {}
};

WebScheduler::~WebScheduler()
{
    for (const auto& handle : m_scheduled) {
        if (!handle.expired()) {
            handle.lock()->cancel();
        }
    }

    m_scheduled.clear();
}

bool isExpired(const std::weak_ptr<WebTask>& handle)
{
    return handle.expired();
}

std::shared_ptr<Cancellable> WebScheduler::schedule(Action action, Scheduler::Microseconds time, bool repeating)
{
    using namespace std::chrono;
    using ms = duration<int, std::milli>;
    const std::chrono::milliseconds MinWebTimeout(4); // minimum schedulable time for setTimeout (see MDN)

    if (time <= MinWebTimeout) {
        action();
        return std::make_shared<NoopCancel>();
    }

    // "Garbage collect" any expired tasks
    auto& s = m_scheduled;
    s.erase(std::remove_if(s.begin(), s.end(), isExpired), s.end());

    // Schedule and save handle so we can cancel if destroyed
    auto task = std::make_shared<WebTask>(std::move(action));
    emscripten::val::module_property("scheduleTask")(task, duration_cast<ms>(time).count(), repeating);
    m_scheduled.push_back(task);
    return task;
}

void WebScheduler::scheduleAndWait(Action action)
{
    action();
}

using namespace emscripten;
EMSCRIPTEN_BINDINGS(webtask)
{
    class_<WebTask>("WebTask")
        .smart_ptr<std::shared_ptr<WebTask>>("WebTaskPtr")
        .function("call", &WebTask::call)
        .function("setCancel", &WebTask::setCancel);
}

} //namespace twitch
