#include "pch.h"
#include "playercore/platform/windows/WindowsPlatform.hpp"
#include "debug/trace.hpp"
#include "Transform.hpp"
#include "AudioSample.hpp"
#include "AudioRenderer.hpp"
#include "EventNames.inl"

using namespace std::literals;
using namespace Microsoft::WRL;
using namespace twitch::windows;

namespace {
	inline twitch::MediaResult AsMediaResult(HRESULT hr) {
		return SUCCEEDED(hr) ? twitch::MediaResult::Ok : twitch::MediaResult(twitch::MediaResult::Error, hr);
	}
}

std::unique_ptr<AudioRenderer> AudioRenderer::create() {
	OutputDebugStringA("#AudioRenderer::create\n");
	TRACE_INFO("AudioRenderer::create");

	HRESULT hr = S_OK;
	auto p = std::unique_ptr<AudioRenderer>(new (std::nothrow) AudioRenderer(hr));
	if(FAILED(hr)) {
		p.reset();
	}
	return p;
}

AudioRenderer::AudioRenderer(HRESULT& hr) {
	OutputDebugStringA("#AudioRenderer::AudioRenderer\n");
	// Start up Media Foundation.
	hr = MFStartup(MF_VERSION);
	if(SUCCEEDED(hr)) {
		m_state = State::Ready;
	}
}

AudioRenderer::~AudioRenderer() {
	OutputDebugStringA("#AudioRenderer::~AudioRenderer\n");
	if(m_state != State::Initial) {
		// Shut down Media Foundation.
		MFShutdown();
	}
}

twitch::MediaResult AudioRenderer::configure(const twitch::MediaFormat& format) {
	m_nChannels = format.hasInt(MediaFormat::Audio_ChannelCount) ? format.getInt(twitch::MediaFormat::Audio_ChannelCount) : 2;
	m_samplesPerSecond = format.hasInt(MediaFormat::Audio_SampleRate) ? format.getInt(twitch::MediaFormat::Audio_SampleRate) : 48000;
	m_bitsPerSample = format.hasInt(MediaFormat::Audio_BitSize) ? format.getInt(twitch::MediaFormat::Audio_BitSize) : 16;
	TRACE_DEBUG("#AudioRenderer::configure #%d %d/s %dbps", m_nChannels, m_samplesPerSecond, m_bitsPerSample);
	return AsMediaResult(Configure(true));
}

twitch::MediaResult AudioRenderer::render(std::shared_ptr<const twitch::MediaSample> input_) {
	TRACE_DEBUG("#AudioRenderer::render %lf", static_cast<const AudioSample&>(*input_).decodeTime.seconds());
	auto input = static_cast<const AudioSample&>(*input_);
	TRACE_DEBUG("rendered time: %lf", input.decodeTime.seconds());
	auto const decodeTime = input.decodeTime.nanoseconds().count() / 100;
	auto const duration = input.duration.nanoseconds().count() / 100;
	return AsMediaResult(Render(decodeTime, duration, input.buffer));
}

twitch::MediaResult AudioRenderer::getRenderedPresentationTime(twitch::MediaTime& time) {
	MFTIME time_ = 0;
	HRESULT hr = GetTime(time_);
	TRACE_DEBUG("#AudioRenderer::getRenderedPresentationTime %lf", SUCCEEDED(hr) ? MediaTime(time_, 10'000'000).seconds() : -1.0);
	if(SUCCEEDED(hr)) {
		time = MediaTime(time_, 10'000'000);
		TRACE_DEBUG("presentation time: %lf", time.seconds());
	}
	return MediaResult::Ok;
}

twitch::MediaResult AudioRenderer::start() {
	OutputDebugStringA("#AudioRenderer::start\n");
	return AsMediaResult(Configure(false));
}

twitch::MediaResult AudioRenderer::stop() {
	OutputDebugStringA("#AudioRenderer::stop\n");
	return MediaResult::Ok;
}

twitch::MediaResult AudioRenderer::flush() {
	OutputDebugStringA("#AudioRenderer::flush\n");
	return AsMediaResult(Configure(true));
}

twitch::MediaResult AudioRenderer::setPlaybackRate(float playbackRate) {
	OutputDebugStringA("#AudioRenderer::setPlaybackRate\n");
	return AsMediaResult(SetPlaybackRate(playbackRate));
}

twitch::MediaResult AudioRenderer::setVolume(float volume) {
	OutputDebugStringA("#AudioRenderer::setVolume\n");
	return AsMediaResult(SetVolume(volume));
}

HRESULT AudioRenderer::Configure(bool isForced) {
	// If the configuration did not change and play-back has not yet started,
	// don't re-configure unless forced to do so.
	if(!isForced && m_writer && !m_presentationClock) {
		return S_OK;
	}

	// Reset all interfaces.
	m_mediaSink.Reset();
	m_presentationClock.Reset();
	m_writer.Reset();
	m_transform.reset();

	// Create an audio renderer.
	ReturnFailure(MFCreateAudioRenderer(nullptr, &m_mediaSink));

	// Get its one stream's type handler.
	Microsoft::WRL::ComPtr<IMFStreamSink> streamSink;
	ReturnFailure(m_mediaSink->GetStreamSinkByIndex(0, &streamSink));
	ComPtr<IMFMediaTypeHandler> typeHandler;
	ReturnFailure(streamSink->GetMediaTypeHandler(&typeHandler));

	// Create an input type using the parameters.
	ComPtr<IMFMediaType> inputType;
	ReturnFailure(MFCreateMediaType(&inputType));
	WAVEFORMATEX waveFormat{};
	waveFormat.wFormatTag = WAVE_FORMAT_PCM;
	waveFormat.nChannels = static_cast<decltype(waveFormat.nChannels)>(m_nChannels);
	waveFormat.nSamplesPerSec = m_samplesPerSecond;
	waveFormat.nAvgBytesPerSec = m_nChannels * m_samplesPerSecond * m_bitsPerSample / 8;
	waveFormat.nBlockAlign = static_cast<decltype(waveFormat.nBlockAlign)>(m_nChannels * m_bitsPerSample / 8);
	waveFormat.wBitsPerSample = static_cast<decltype(waveFormat.wBitsPerSample)>(m_bitsPerSample);
	ReturnFailure(MFInitMediaTypeFromWaveFormatEx(inputType.Get(), &waveFormat, sizeof(waveFormat)));

	// Determine if the audio renderer's stream supports that type.
	ComPtr<IMFMediaType> supportedType;
	if(SUCCEEDED(typeHandler->IsMediaTypeSupported(inputType.Get(), &supportedType))) {
		// It does; set that type as its type.
		ReturnFailure(typeHandler->SetCurrentMediaType(inputType.Get()));
	} else {
		// It does not; create a transform with that type and set the transform's
		// output type as its type.
		ReturnFailure(Transform::Create(inputType.Get(), m_transform));
		ReturnFailure(typeHandler->SetCurrentMediaType(m_transform->MediaType.Get()));
		inputType = m_transform->MediaType;
	}

	// Determine the slowest and fastest rate for the renderer.
	ComPtr<IMFGetService> getService;
	ReturnFailure(m_mediaSink.CopyTo(IID_PPV_ARGS(&getService)));
	ComPtr<IMFRateSupport> rs;
	ReturnFailure(getService->GetService(MF_RATE_CONTROL_SERVICE, IID_PPV_ARGS(&rs)));
	ReturnFailure(rs->GetSlowestRate(MFRATE_FORWARD, FALSE, &m_slowestRate));
	ReturnFailure(rs->GetFastestRate(MFRATE_FORWARD, FALSE, &m_fastestRate));

	// Create and start a sink writer.
	ReturnFailure(MFCreateSinkWriterFromMediaSink(m_mediaSink.Get(), nullptr, &m_writer));
	ReturnFailure(m_writer->SetInputMediaType(0, inputType.Get(), nullptr));
	ReturnFailure(m_writer->BeginWriting());

	return S_OK;
}

HRESULT AudioRenderer::Render(LONGLONG time, LONGLONG duration, std::vector<uint8_t> const& data) {
	ComPtr<IMFMediaBuffer> buffer;
	LPBYTE p = nullptr;
	ComPtr<IMFSample> sample;
	auto size = static_cast<DWORD>(data.size());
	ReturnFailure(MFCreateMemoryBuffer(size, &buffer));
	ReturnFailure(buffer->Lock(&p, nullptr, nullptr));
	memcpy(p, data.data(), data.size());
	ReturnFailure(buffer->Unlock());
	ReturnFailure(buffer->SetCurrentLength(size));
	ReturnFailure(MFCreateSample(&sample));
	ReturnFailure(sample->AddBuffer(buffer.Get()));
	ReturnFailure(sample->SetSampleTime(time));
	ReturnFailure(sample->SetSampleDuration(duration));
	if(!m_presentationClock) {
		ReturnFailure(m_mediaSink->GetPresentationClock(&m_presentationClock));
		LONGLONG sampleTime;
		ReturnFailure(sample->GetSampleTime(&sampleTime));
		ReturnFailure(m_presentationClock->Start(sampleTime));
	}
	if(m_transform) {
		ReturnFailure(m_transform->Write(sample.Get(), m_writer.Get()));
	} else {
		ReturnFailure(m_writer->WriteSample(0, sample.Get()));
	}
	return S_OK;
}

HRESULT AudioRenderer::SetPlaybackRate(float playbackRate) {
	if(!m_mediaSink) {
		return MF_E_INVALIDREQUEST;
	}
	if(playbackRate < m_slowestRate || playbackRate > m_fastestRate) {
		return E_INVALIDARG;
	}
	m_playbackRate = playbackRate;
	return S_OK;
}

HRESULT AudioRenderer::SetVolume(float volume) {
	if(!m_mediaSink) {
		return MF_E_INVALIDREQUEST;
	}
	if(volume < 0.0f || volume > 1.0f) {
		return E_INVALIDARG;
	}
	m_volume = volume;
	return S_OK;
}

HRESULT AudioRenderer::GetTime(MFTIME& time) {
	if(m_presentationClock == nullptr) {
		return E_POINTER;
	}
	ReturnFailure(m_presentationClock->GetTime(&time));
	return S_OK;
}

/*
HRESULT AudioRenderer::InternalSetPlaybackRate() {
	ComPtr<IMFRateSupport> rateSupport;
	ComPtr<IMFRateControl> rateControl;
	ReturnFailure(MFGetService(m_session.Get(), MF_RATE_CONTROL_SERVICE, IID_PPV_ARGS(&rateSupport)));
	float actualRate;
	ReturnFailure(rateSupport->IsRateSupported(FALSE, m_playbackRate, &actualRate));
	ReturnFailure(MFGetService(m_session.Get(), MF_RATE_CONTROL_SERVICE, IID_PPV_ARGS(&rateControl)));
	ReturnFailure(rateControl->SetRate(FALSE, actualRate));
	return S_OK;
}
*/

HRESULT AudioRenderer::InternalSetVolume() {
	ComPtr<IMFSimpleAudioVolume> audioVolume;
	ReturnFailure(MFGetService(m_mediaSink.Get(), MR_POLICY_VOLUME_SERVICE, IID_PPV_ARGS(&audioVolume)));
	ReturnFailure(audioVolume->SetMasterVolume(m_volume));
	return S_OK;
}
