#pragma once

#include "playercore/platform/HttpClient.hpp"
#include "util/Concurrency.hpp"
#include "debug/trace.hpp"

// Set this to 1 to test different http timer failures at predetermined sync points
#define HTTPTIMER_SIMULATE_TIMEOUT 0

namespace twitch {
namespace windows {

    class HttpTimer {
    public:
        template<typename DURATION>
        HttpTimer(DURATION timeout, int syncPoint)
            : m_callback([this](int) { onCallback(); })
            , m_originalTimeoutMs(std::chrono::duration_cast<std::chrono::milliseconds>(timeout).count())
            , m_timer(std::unique_ptr<concurrency::timer<int>>(new concurrency::timer<int>(static_cast<unsigned int>(std::chrono::duration_cast<std::chrono::milliseconds>(timeout).count()), 0, &m_callback, false)))
            , m_syncPoint(syncPoint)
            , m_lastSyncPointReached(-1)
        {
        }
        ~HttpTimer();

        HttpTimer(const HttpTimer& rhs) = delete;
        HttpTimer& operator=(const HttpTimer& rhs) = delete;

        /// Start the timer, if it was not running
        void start();

        /// Stop the timer, if it was running
        void stop();

        /// Restart the timer, if it has not timed out yet. The timer will
        /// restart with the original timeout value.
        void restart();

        void forceTimeout();
        void syncPoint(int syncPoint);

        void setTimeoutFunction(std::function<void()>&& onTimeout);
        bool hasTimedOut();

    private:
        void onCallback();

    private:
        concurrency::call<int> m_callback;
        std::unique_ptr<concurrency::timer<int>> m_timer;
        std::atomic<bool> m_timeout = false;
        std::chrono::milliseconds m_originalTimeoutMs;
        std::function<void()> m_onTimeout;

        const int m_syncPoint;
        int m_lastSyncPointReached;

        RecursiveMutex m_mutex;
    };

}
}
