#include "pch.h"
#include "playercore/platform/windows/WindowsPlatform.hpp"
#include "debug/trace.hpp"
#include "Transform.hpp"

using namespace Microsoft::WRL;
using namespace twitch::windows;

namespace {
	HRESULT MakeSample(LONGLONG time, LONGLONG duration, DWORD size, ComPtr<IMFSample>& sample) {
		ComPtr<IMFMediaBuffer> buffer;
		ReturnFailure(MFCreateMemoryBuffer(size, &buffer));
		ReturnFailure(buffer->SetCurrentLength(0));
		ReturnFailure(MFCreateSample(&sample));
		ReturnFailure(sample->AddBuffer(buffer.Get()));
		if(time) {
			ReturnFailure(sample->SetSampleTime(time));
		}
		if(duration) {
			ReturnFailure(sample->SetSampleDuration(duration));
		}
		return S_OK;
	}
}

HRESULT Transform::Create(IMFMediaType* mediaType, std::unique_ptr<Transform>& p) {
	HRESULT hr = S_OK;
	p.reset(new (std::nothrow) Transform(mediaType, hr));
	if(p == nullptr) {
		hr = E_OUTOFMEMORY;
	} else if(FAILED(hr)) {
		p.reset();
	}
	return hr;
}

HRESULT Transform::Finish(IMFSinkWriter* writer) {
	ReturnFailure(m_transform->ProcessMessage(MFT_MESSAGE_NOTIFY_END_OF_STREAM, 0));
	ReturnFailure(m_transform->ProcessMessage(MFT_MESSAGE_COMMAND_DRAIN, 0));
	ReturnFailure(WriteTransformedSamples(writer));
	ReturnFailure(m_transform->ProcessMessage(MFT_MESSAGE_NOTIFY_END_STREAMING, 0));
	return S_OK;
}

HRESULT Transform::Write(IMFSample* sample, IMFSinkWriter* writer) {
	ReturnFailure(m_transform->ProcessInput(0, sample, 0));
	ReturnFailure(WriteTransformedSamples(writer));
	return S_OK;
}

Transform::Transform(IMFMediaType* mediaType, HRESULT& hr) {
	hr = [this, mediaType] {
		// Create re-sampler transform.
		ReturnFailure(CoCreateInstance(CLSID_CResamplerMediaObject, nullptr, CLSCTX_INPROC_SERVER, IID_PPV_ARGS(&m_transform)));

		// Set maximum quality filtering.
		ComPtr<IWMResamplerProps> resamplerProps;
		ReturnFailure(m_transform.CopyTo(IID_PPV_ARGS(&resamplerProps)));
		ReturnFailure(resamplerProps->SetHalfFilterLength(60));

		// Set the re-sampler transform's input type.
		ReturnFailure(m_transform->SetInputType(0, mediaType, 0));

		// Set the re-sampler transform's output type.
		ReturnFailure(MFCreateMediaType(&m_mediaType));
		ReturnFailure(m_mediaType->SetGUID(MF_MT_MAJOR_TYPE, MFMediaType_Audio));
		ReturnFailure(m_mediaType->SetGUID(MF_MT_SUBTYPE, MFAudioFormat_PCM));
		ReturnFailure(m_mediaType->SetUINT32(MF_MT_AUDIO_NUM_CHANNELS, 2));
		ReturnFailure(m_mediaType->SetUINT32(MF_MT_AUDIO_SAMPLES_PER_SECOND, 48000));
		ReturnFailure(m_mediaType->SetUINT32(MF_MT_AUDIO_BLOCK_ALIGNMENT, 4));
		ReturnFailure(m_mediaType->SetUINT32(MF_MT_AUDIO_AVG_BYTES_PER_SECOND, 48000 * 4));
		ReturnFailure(m_mediaType->SetUINT32(MF_MT_AUDIO_BITS_PER_SAMPLE, 16));
		ReturnFailure(m_mediaType->SetUINT32(MF_MT_ALL_SAMPLES_INDEPENDENT, TRUE));
		ReturnFailure(m_transform->SetOutputType(0, m_mediaType.Get(), 0));

		// Save output stream information.  The WriteTransformedSamples method
		// uses it to determine if it needs to create samples.
		ReturnFailure(m_transform->GetOutputStreamInfo(0, &m_info));

		// Start the re-sampler transform.
		ReturnFailure(m_transform->ProcessMessage(MFT_MESSAGE_COMMAND_FLUSH, 0));
		ReturnFailure(m_transform->ProcessMessage(MFT_MESSAGE_NOTIFY_BEGIN_STREAMING, 0));
		ReturnFailure(m_transform->ProcessMessage(MFT_MESSAGE_NOTIFY_START_OF_STREAM, 0));
		return S_OK;
	}();
}

HRESULT Transform::WriteTransformedSamples(IMFSinkWriter* writer) {
	for(;;) {
		ComPtr<IMFSample> sample;
		MFT_OUTPUT_DATA_BUFFER odb{};
		if((m_info.dwFlags & MFT_OUTPUT_STREAM_PROVIDES_SAMPLES) == 0) {
			ReturnFailure(MakeSample(0, 0, 16384, sample));
			odb.pSample = sample.Detach();
		}
		DWORD status;
		HRESULT hr = m_transform->ProcessOutput(0, 1, &odb, &status);
		if(odb.pEvents) {
			DebugBreak();
			odb.pEvents->Release();
		}
		sample.Attach(odb.pSample);
		if(hr == MF_E_TRANSFORM_NEED_MORE_INPUT) {
			break;
		}
		ReturnFailure(hr);
		ReturnFailure(writer->WriteSample(0, odb.pSample));
	}
	return S_OK;
}
