#pragma once

#include "MediaFoundation.hpp"
#include "playercore/MediaDecoder.hpp"
#include "VideoSample.hpp"
#include "CoInitialize.hpp"

namespace twitch {
namespace windows {
        class VideoDecoder : public MediaFoundation, public CCoInitialize, public MediaDecoder {
        public:

       // https://docs.microsoft.com/en-us/windows/desktop/medfound/h-264-video-decoder
#ifdef __cplusplus_winrt
        // TODO: Use https://docs.microsoft.com/en-us/windows/desktop/medfound/codecapi-avdecvideomaxcodedheight
        // and https://docs.microsoft.com/en-us/windows/desktop/medfound/codecapi-avdecvideomaxcodedwidth
        static const int MaxDecodeWidth = 4096;
        static const int MaxDecodeHeight = 2304;
#else
        // https://docs.microsoft.com/en-us/windows/desktop/medfound/h-264-video-decoder#transform-attributes
        static const int MaxDecodeWidth = 1920;
        static const int MaxDecodeHeight = 1088;
#endif
        enum ProcessType {
            Default = 0,
            Discard = 1,
        };

        VideoDecoder(Microsoft::WRL::ComPtr<ID3D11Device> device,
                     Microsoft::WRL::ComPtr<IMFDXGIDeviceManager> pDXGIManager,
                     bool isRunningOnXBox);

        VideoDecoder(const VideoDecoder& rhs) = delete;
        VideoDecoder& operator=(const VideoDecoder& rhs) = delete;

        MediaResult configure(const MediaFormat& input, MediaFormat& output) override;
        MediaResult decode(const twitch::MediaSampleBuffer& input) override;
        MediaResult getOutput(std::shared_ptr<twitch::MediaSample>& output) override;
        MediaResult hasOutput(bool& hasOutput) override;
        MediaResult flush() override;
        MediaResult reset() override;

        static const int TexturePaddingSize = 16;
    private:
        MediaResult configureSoftware(const MediaFormat& input);
        MediaResult processInputSoftware(const twitch::MediaSampleBuffer& input);
        MediaResult processOutputSoftware(HRESULT& hr, ProcessType pt = ProcessType::Default);

        void requestNewSample();
        void handleOutputDataBuffer(MFT_OUTPUT_DATA_BUFFER& outputDataBuffer, ProcessType pt);
        void associateOutput(IMFSample* sample);

        bool fallbackInSoftwareModeIfFailed(HRESULT hr, const char * message);
        MediaResult configureHardware();
        MediaResult processInputHardware(const twitch::MediaSampleBuffer& input);
        MediaResult processOutputHardware();

        // outputs
        std::list<std::shared_ptr<VideoSample>> m_outputSamples;

        // H.264 Decoder
        Microsoft::WRL::ComPtr<IMFTransform> m_decoderTransform;
        Microsoft::WRL::ComPtr<IMFMediaType> m_inputMediaType;
        Microsoft::WRL::ComPtr<IMFMediaType> m_outputMediaType;
        Microsoft::WRL::ComPtr<ID3D11Device> m_d3dDevice;
        Microsoft::WRL::ComPtr<IMFDXGIDeviceManager> m_dxgiManager;
        Microsoft::WRL::ComPtr<ID3D11VideoDevice> m_videoService;
        Microsoft::WRL::ComPtr<ID3D11VideoContext> m_videoContext;
        Microsoft::WRL::ComPtr<ID3D11VideoDecoder> m_decoder;
        GUID m_decoderGUID;

        int m_currentSampleWidth = 0;
        int m_currentSampleHeight = 0;

        struct FrameBuffer
        {
            IMFSample* sample = nullptr;

            ~FrameBuffer()
            {
                if (sample != nullptr) {
                    sample->Release();
                    sample = nullptr;
                }
            }
        };

        static const int FrameBuffersCount = 16;
        int m_currentFrameBufferIndex = 0;
        FrameBuffer m_frameBuffers[FrameBuffersCount];
        IMFSample * m_currentSample = nullptr;

        bool m_softwareDecoder = true;
        bool m_isRunningOnXBox = true;

        MFT_INPUT_STREAM_INFO m_inputStreamInfo;
        MFT_OUTPUT_STREAM_INFO m_outputStreamInfo;

        // Specifies the maximum number of output samples that a Microsoft Media Foundation transform (MFT) will have outstanding in the pipeline at any time.
        static const int MaxWorkerSamples = 30;
        static const int MaxOutputSamples = 24;
        int m_currentSampleIndex = 0;
        Microsoft::WRL::ComPtr<IMFSample> m_samples[MaxOutputSamples];
    };
}
}
