#include "pch.h"
#include "VideoRenderer.hpp"
#include "VideoDecoder.hpp"
#include "VideoSample.hpp"
#include "debug/trace.hpp"
#include "playercore/platform/windows/PlayerCore.hpp"

namespace twitch {
namespace windows {
VideoRenderer::VideoRenderer(WindowsPlatform::OnRenderFrame onRenderFrame, const ReferenceClock& clock)
    : m_onRenderFrame(onRenderFrame)
    , m_clock(clock)
{
}

VideoRenderer::~VideoRenderer()
{
    TRACE_DEBUG("VideoRenderer::DTor()");
}

MediaResult VideoRenderer::configure(const MediaFormat& format)
{
    if (!m_onRenderFrame) {
        TRACE_DEBUG("VideoRenderer no m_onRenderFrame callback set");
        return MediaResult::ErrorInvalidState;
    }
    m_width = format.getInt(MediaFormat::Video_Width);
    m_height = format.getInt(MediaFormat::Video_Height);

    return MediaResult::Ok;
}

MediaResult VideoRenderer::render(std::shared_ptr<const twitch::MediaSample> input)
{
    const VideoSample& videoSample = static_cast<const VideoSample&>(*input);

    MediaTime mediaTime = m_clock.getMediaTime();
    if (!mediaTime.valid()) {
        mediaTime = MediaTime::zero();
    }
    MediaTime delta = videoSample.presentationTime - mediaTime;

    //TRACE_DEBUG("VideoRenderer::render at ps %f and clock at %f", videoSample.presentationTime.seconds(), mediaTime.seconds());

    const MediaTime maxDelta(1.0);
    if (delta > maxDelta) {
        delta = MediaTime::zero();
    }

    if (delta.milliseconds().count() > 0.0) {
        std::this_thread::sleep_for(delta.milliseconds());
    }

    if (m_onRenderFrame) {
        Microsoft::WRL::ComPtr<ID3D11Texture2D> texture = PlayerCore::getTextureFromVideoSample(videoSample);

        D3D11_TEXTURE2D_DESC desc;
        texture->GetDesc(&desc);

        const int textureDelta = (std::abs(static_cast<int>(desc.Width - m_width)) + std::abs(static_cast<int>(desc.Height - m_height)));
        const bool isTextureValid = textureDelta <= (2 * VideoDecoder::TexturePaddingSize);
        if (isTextureValid) {
            m_onRenderFrame(videoSample, m_width, m_height);
        }
    }

    m_renderedFrames++;
    m_lastRenderedPresentationTime = videoSample.presentationTime;

    return MediaResult::Ok;
}

MediaResult VideoRenderer::getDroppedFrames(int& count)
{
    count = m_droppedFrames;
    return MediaResult::Ok;
}

MediaResult VideoRenderer::getRenderedFrames(int& count)
{
    count = m_renderedFrames;
    return MediaResult::Ok;
}

MediaResult VideoRenderer::getRenderedPresentationTime(MediaTime& time)
{
    time = m_lastRenderedPresentationTime;
    return MediaResult::Ok;
}

MediaResult VideoRenderer::start()
{
    return MediaResult::Ok;
}

MediaResult VideoRenderer::stop()
{
    TRACE_DEBUG("VideoRenderer::stop() called.");
    return MediaResult::Ok;
}

MediaResult VideoRenderer::flush()
{
    m_droppedFrames = 0;
    m_renderedFrames = 0;
    return MediaResult::Ok;
}

MediaResult VideoRenderer::setPlaybackRate(float rate)
{
    (void)rate;
    return MediaResult::Ok;
}

}
}
