#include "pch.h"
#include "playercore/platform/windows/WindowsPlatform.hpp"
#include "AudioDecoder.hpp"
#ifdef __cplusplus_winrt
# include "platforms/uwp/AudioRenderer.hpp"
# include "platforms/uwp/HttpClient.hpp"
#else
# include "platforms/windows/AudioRenderer.hpp"
# include "platforms/windows/HttpClient.hpp"
#endif
#include "Log.hpp"
#include "debug/trace.hpp"
#include "checkhr.hpp"
#include "VideoDecoder.hpp"
#include "VideoRenderer.hpp"

using namespace Microsoft::WRL;
using namespace twitch;

#ifdef __cplusplus_winrt
# define AudioRenderer uwp::AudioRenderer
# define WinHttpClient uwp::HttpClient

namespace
{
    bool isRunningOnXbox()
    {
        static bool isChecked = false;
        static bool isRunningOnXbox = false;
        if (isChecked) {
            return isRunningOnXbox;
        }

        auto versionInfoDeviceFamily = Windows::System::Profile::AnalyticsInfo::VersionInfo->DeviceFamily;
        if (versionInfoDeviceFamily != nullptr) {
            std::wstring strDeviceFamily = versionInfoDeviceFamily->Data();
            std::transform(strDeviceFamily.begin(), strDeviceFamily.end(), strDeviceFamily.begin(), std::towlower);
            if (strDeviceFamily.find(L"xbox") != std::wstring::npos) {
                isRunningOnXbox = true;
            }
        }

        isChecked = true;
        return isRunningOnXbox;
    }
}

const std::string WindowsPlatform::m_name = "UWP";
#else
# define AudioRenderer windows::AudioRenderer
# define WinHttpClient windows::HttpClient

namespace
{
    constexpr bool isRunningOnXbox() { return false; }
}

const std::string WindowsPlatform::m_name = "Win64";
#endif

std::unique_ptr<MediaDecoder> WindowsPlatform::createDecoder(std::shared_ptr<const MediaFormat> format)
{
    if (format->getType().matches(MediaType::Audio_AAC)) {
        return std::unique_ptr<MediaDecoder>(new windows::AudioDecoder());
    } else if (format->getType().matches(MediaType::Video_AVC)) {
        return std::unique_ptr<MediaDecoder>(new windows::VideoDecoder(m_device, m_dxgiManager, isRunningOnXbox()));
    } else {
        return nullptr;
    }
}

std::unique_ptr<MediaRenderer> WindowsPlatform::createRenderer(const ReferenceClock& clock, std::shared_ptr<const MediaFormat> format)
{
    if (format->getType().matches(MediaType::Audio_AAC)) {
        return AudioRenderer::create();
    } else if (format->getType().matches(MediaType::Video_AVC)) {
        return std::unique_ptr<MediaRenderer>(new windows::VideoRenderer(m_onRenderFrame, clock));
    } else {
        return nullptr;
    }
}

void WindowsPlatform::hError(const std::string& message, HRESULT errorCode)
{
    TRACE_ERROR("%s with HRESULT: 0x%08x", message.c_str(), errorCode);
}

std::string wstringTostring(const std::wstring& inputString)
{
    std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>> converter;
    std::string narrow = converter.to_bytes(inputString);

    return narrow;
}

WindowsPlatform::WindowsPlatform(ComPtr<ID3D11Device> device, OnRenderFrame onRenderFrame)
    : m_capabilities()
    , m_log(std::make_shared<windows::PrintLog>())
    , m_httpClient(std::make_shared<WinHttpClient>())
    , m_onRenderFrame(onRenderFrame)
    , m_device(device)
{
#ifdef __cplusplus_winrt
    ::Platform::String ^ tempFolder = Windows::Storage::ApplicationData::Current->TemporaryFolder->Path;
    std::wstring tempFolderW(tempFolder->Begin());
    std::string tempFolderA(tempFolderW.begin(), tempFolderW.end());

    Windows::Storage::ApplicationData::Current->LocalFolder->CreateFolderAsync("tmp");
    m_tempPath = tempFolderA + "/tmp";
#else
    size_t size;
    getenv_s(&size, nullptr, 0, "TEMP");
    m_tempPath.resize(size);
    getenv_s(&size, &m_tempPath[0], size, "TEMP");
    m_tempPath.back() = '\\';
    m_tempPath += "player-core-";
    GUID guid;
    CoCreateGuid(&guid);
    OLECHAR sz[44];
    auto n = StringFromGUID2(guid, sz, _countof(sz));
    m_tempPath += std::string(sz + 1, sz + n - 2);
    _mkdir(m_tempPath.c_str());
#endif

    SetupGraphics();

    m_capabilities.supportsLowLatencyABR = true;
}

VideoDecoderCapabilities WindowsPlatform::getVideoDecoderCapabilities(const MediaType& mediaType)
{
    VideoDecoderCapabilities capabilities;
    if (mediaType.matches(MediaType::Video_AVC)) {
        // "The Media Foundation H.264 video decoder is a Media Foundation Transform that supports decoding of Baseline, Main, and High profiles, up to level 5.1."
        //  *The comment on the website is maybe not up to date*
        // [https://docs.microsoft.com/en-us/windows/desktop/medfound/h-264-video-decoder]
        capabilities.maxWidth = windows::VideoDecoder::MaxDecodeWidth;
        capabilities.maxHeight = windows::VideoDecoder::MaxDecodeHeight;
        capabilities.maxLevel = eAVEncH264VLevel5_1;
        capabilities.maxProfile = eAVEncH264VProfile_High;
    }

    return capabilities;
}

void WindowsPlatform::setCurrentThreadName(const std::string& name)
{
    std::wstring wide(name.length(), L' ');
    std::copy(name.begin(), name.end(), wide.begin());

    SetThreadDescription(GetCurrentThread(), wide.c_str());
}

std::shared_ptr<Log> WindowsPlatform::getLog() const
{
    return m_log;
}

std::shared_ptr<twitch::HttpClient> WindowsPlatform::getHttpClient() const
{
    return m_httpClient;
}

const std::string& WindowsPlatform::getName() const
{
    return m_name;
}

const std::string& WindowsPlatform::getTempPath() const
{
    return m_tempPath;
}

const Capabilities& WindowsPlatform::getCapabilities() const
{
    return m_capabilities;
}

MediaResult WindowsPlatform::SetupGraphics()
{
    assert(m_device);

    CHECK_HR(MFCreateDXGIDeviceManager(&m_deviceResetToken, &m_dxgiManager), "MFCreateDXGIDeviceManager failed");
    CHECK_HR(m_dxgiManager->ResetDevice(m_device.Get(), m_deviceResetToken), "ResetDevice failed");

    ComPtr<ID3D10Multithread> multithread;
    CHECK_HR(m_device.CopyTo(IID_PPV_ARGS(&multithread)), "QueryInterface Multithread failed");
    multithread->SetMultithreadProtected(true);

    ComPtr<ID3D11DeviceContext> immediateContext;
    m_device->GetImmediateContext(&immediateContext);
    CHECK_HR(immediateContext.CopyTo(IID_PPV_ARGS(&multithread)), "QueryInterface ImmediateContext Multithread failed");
    multithread->SetMultithreadProtected(true);

    return MediaResult::Ok;
}
