'''
Devicefarm Adaptor
'''
import core
import time

class DeviceFarmAdaptor:
    packageFolder = 'devicefarmPackage'
    resultsFolder = 'devicefarmResults'
    
    ''' Constructor '''
    def __init__(self, config, platform):
        self.platform = platform
        core.log.message('Creating Devicefarm adaptor')
        core.log.message('- Platform: '+str(self.platform))
        core.log.message('- Project: '+config.jobSetting('devicefarm.project'))
        core.log.message('- DevicePool: '+config.jobSetting('devicefarm.devicePool'))
        self.boto3 = config.jobSetting('aws.boto3')
    
    ''' Execution Dispatcher '''
    def execute(self, config, resolveResultXML=True):
        result = 0
        if self.platform == core.const.Platform.iOS:
            result = self.executeIOS(config)
        elif self.platform == core.const.Platform.iOSWeb:
            result = self.executeIOSWeb(config)
        elif self.platform == core.const.Platform.Android:
            result = self.executeAndroid(config)
        elif self.platform == core.const.Platform.AndroidWeb:
            result = self.executeAndroidWeb(config)
        else:
            core.log.error('Unsupported platform: '+str(self.platform))
        if resolveResultXML:
            core.util.findResultXML(core.shell.join_path(config.getOutDir(), self.resultsFolder))
        return result

    ''' Execute testing on iOS platform '''
    def executeIOS(self, config):
        # Create iOS package as outline in: https://docs.aws.amazon.com/devicefarm/latest/developerguide/test-types-ios-xctest.html
      
        # Find Application, create package directory
        core.log.message('Packaging build for AWS DeviceFarm...')
        paths = core.shell.find('*.app', config.getWorkDir(), findFiles=False)
        if len(paths) > 1:
            core.log.warning('Found multiple applications, selecting first')
        appPath=paths[0]
        core.log.message('Processing application at "'+appPath+'"...')
        packageDir=core.shell.join_path(config.getOutDir(), self.packageFolder)
        core.shell.mkdir(packageDir, deleteIfExists=True)
        
        # Create Application Package
        payloadDir=core.shell.join_path(packageDir, 'Payload')
        path, file = core.shell.path_split(appPath)
        core.shell.copytree(appPath, core.shell.join_path(payloadDir, file) )
        appName, appExt = core.shell.filename_split(file)
        ipaPath = core.shell.join_path(packageDir, appName + '.ipa')
        core.shell.zip(payloadDir, ipaPath)
        core.shell.remove(payloadDir)
        
        # Create XCTest Package and publish
        paths = core.shell.find('*.xctest', appPath, findFiles=False)
        if len(paths) > 1:
            core.log.warning('Found multiple directories, selecting first')
        xcPath=paths[0]
        targetDir=core.shell.join_path(packageDir, appName + '.xctest')
        core.shell.copytree(xcPath, targetDir)
        core.shell.zip(targetDir)
        core.shell.remove(targetDir)
        
        # Upload package (.IPA and .XCTEST.ZIP)
        projectArn = self.getProjectArn(name=config.jobSetting('devicefarm.project'))
        appArn = self.initiateUpload(projectArn=projectArn, name=ipaPath, type='IOS_APP')
        if not self.checkUploadStatus(uploadArn=appArn): return 1
        driverArn = self.initiateUpload(projectArn=projectArn, name=targetDir + '.zip', type='XCTEST_TEST_PACKAGE')
        if not self.checkUploadStatus(uploadArn=driverArn): return 1
    
        # Schedule devicefarm run
        devicePoolArn = self.getDevicepoolArn(name = config.jobSetting('devicefarm.devicePool'), projectArn=projectArn)
        runArn = self.scheduleRun(projectArn=projectArn, devicePoolArn=devicePoolArn, appArn=appArn, test={'type': 'XCTEST', 'testPackageArn': driverArn}, configuration={'customerArtifactPaths': {'iosPaths': ['Documents']}} )
        runResult = self.checkRunStatus(runArn=runArn, timeoutSec=config.getJobTimeoutSec())
        if runResult != 'PASSED':
            core.log.warning('Devicefarm run did not pass, attempting to gather results...')
        
        # Download results
        self.fetchResults(runArn=runArn, downloadFolder=core.shell.join_path(config.getOutDir(), self.resultsFolder))
        return runResult == 'PASSED'
            
    ''' Execute testing on iOS Web platform '''
    def executeIOSWeb(self, config):
        core.log.error('Not implemented')
        return 0
    
    ''' Execute testing on Android platform '''
    def executeAndroid(self, config):
        # Test artifact location
        appApkPath = core.shell.os_path(config.getOutDir()+'/testApp-debug.apk')
        testApkPath = core.shell.os_path(config.getOutDir()+'/testApp-debug-androidTest.apk')
        
        # Upload package (app and test APK)
        projectArn = self.getProjectArn(name=config.jobSetting('devicefarm.project'))
        appArn = self.initiateUpload(projectArn=projectArn, name=appApkPath, type='ANDROID_APP')
        if not self.checkUploadStatus(uploadArn=appArn): return 1
        driverArn = self.initiateUpload(projectArn=projectArn, name=testApkPath, type='INSTRUMENTATION_TEST_PACKAGE')
        if not self.checkUploadStatus(uploadArn=driverArn): return 1
        
        # Schedule devicefarm run
        devicePoolArn = self.getDevicepoolArn(name = config.jobSetting('devicefarm.devicePool'), projectArn=projectArn)
        runArn = self.scheduleRun(projectArn=projectArn, devicePoolArn=devicePoolArn, appArn=appArn, test={'type': 'INSTRUMENTATION', 'testPackageArn': driverArn}, configuration={'customerArtifactPaths': {'androidPaths': ['/sdcard/']}} )
        runResult = self.checkRunStatus(runArn=runArn, timeoutSec=config.getJobTimeoutSec())
        if runResult != 'PASSED':
            core.log.warning('Devicefarm run did not pass, attempting to gather results...')
        
        # Download results
        self.fetchResults(runArn=runArn, downloadFolder=core.shell.join_path(config.getOutDir(), self.resultsFolder))
        return runResult == 'PASSED'

    ''' Execute testing on Android Web platform '''
    def executeAndroidWeb(self, config):
        core.log.error('Not implemented')
        return 0

    ''' Return ARN from a list of objects by name '''
    def getArnFromList(self, list, name):
        for i in range(0, len(list)):
            if list[i]['name'] == name:
                return list[i]['arn']
        return None

    ''' Return ARN of given devicefarm project '''
    def getProjectArn(self, name):
        core.log.important('Retrieving project ARN for "'+name+'"...' )
        list = self.boto3.client.list_projects()['projects']
        arn = self.getArnFromList(list, name)
        if not arn:
            core.log.error('Devicefarm project "'+name+'" not found')
        core.log.message('Project ARN is: '+arn)
        return arn

    ''' Return ARN of given devicefarm device pool '''
    def getDevicepoolArn(self, name, projectArn):
        core.log.important('Retrieving device pool ARN for "'+name+'"')
        list = self.boto3.client.list_device_pools(arn=projectArn)['devicePools']
        arn = self.getArnFromList(list, name)
        if not arn:
            core.log.error('Devicepool "'+name+'" not found')
        core.log.message('DevicePool ARN is: '+arn)
        return arn

    ''' Initiate async upload '''
    def initiateUpload(self, projectArn, name, type):
        core.log.important('Uploading "'+name+'" ('+type+')...')
        path, file = core.shell.path_split(name)
        descriptor = self.boto3.client.create_upload(projectArn=projectArn, name=file, type=type )
        if not descriptor:
            core.log.error('Unable to create upload (boto3.create_upload() failed)')
        uploadUrl = descriptor['upload']['url']
        core.http.uploadFile(name, uploadUrl, headers={'content-type': 'application/octet-stream'})
        uploadArn = descriptor['upload']['arn']
        return uploadArn

    ''' Check upload status within given time limit '''
    def checkUploadStatus(self, uploadArn, timeoutSec=60*1):
        core.log.important('Checking upload status of "'+uploadArn+'", with timeout of "'+str(timeoutSec)+'" seconds...')
        timeMax = time.time() + timeoutSec
        while time.time() < timeMax:
            descriptor = self.boto3.client.get_upload(arn=uploadArn)['upload']
            if not descriptor:
                core.log.error('Unable to retrieve upload status (boto3.get_upload() failed)')
            status = descriptor['status']
            if status == 'SUCCEEDED':
                core.log.message('Upload completed successfully')
                return True
            elif status == 'FAILED':
                metadata = descriptor['metadata']
                core.log.error('Upload to devicefarm failed: ' + str(metadata))
            time.sleep(5)
        core.log.error('Upload to devicefarm timed out')
        return False
                
    ''' Schedule a run on devicefarm '''
    def scheduleRun(self, projectArn, devicePoolArn, appArn, test, configuration={}):
        core.log.message('Scheduling run on devicefarm...', core.log.TextFormat.BOLD)
        descriptor = self.boto3.client.schedule_run(projectArn=projectArn, appArn=appArn, devicePoolArn=devicePoolArn, test=test, configuration=configuration)
        if not descriptor:
            core.log.error('Unable to schedule run (boto3.schedule_run() failed')
        runArn = descriptor['run']['arn']
        return runArn

    ''' Check run status within given timelimit '''
    def checkRunStatus(self, runArn, timeoutSec):
        core.log.important('Checking run status of "'+runArn+'", with timeout of "'+str(timeoutSec)+'" seconds...')
        timeMax = time.time() + timeoutSec
        lastStatus='IDLE'
        while time.time() < timeMax:
            descriptor = self.boto3.client.get_run(arn=runArn)['run']
            if not descriptor:
                core.log.error('Unable to retrieve run status (boto3.get_run() failed)')
            status = descriptor['status']
            result = descriptor['result']
            if status != lastStatus:
                core.log.message('Status change: '+lastStatus+' => '+status)
                lastStatus=status
            if status == 'COMPLETED':
                core.log.message('Run completed with result: '+result)
                if result == 'ERRORED':
                    core.log.error('Devicefarm run failed: '+str(descriptor))
                return result
            elif status == 'STOPPING':
                # Don't wait for devicefarm in STOPPING state
                break
            time.sleep(5)
        core.log.warning('Run on devicefarm timed out, or was stopped before completion')
        self.stopRun(runArn=runArn)
        return 'STOPPED'
                
    ''' Stop devicefarm run given by ARN '''
    def stopRun(self, runArn):
        core.log.important('Stopping run '+runArn+'...')
        response = self.boto3.client.stop_run(arn=runArn)
        return True
    
    ''' Fetch all results for a given run on devicefarm '''
    def fetchResults(self, runArn, downloadFolder):
        result = self.boto3.client.get_run(arn=runArn)['run']['result']
        if result != 'PENDING':
            downloadFolder = core.shell.join_path(downloadFolder, runArn)
            core.log.message('Fetching devicefarm results for "'+runArn+'" to "'+downloadFolder+'"...')
            jobList = self.boto3.client.list_jobs(arn=runArn)['jobs']
            for job in jobList:
                deviceName = job['device']['name']
                deviceFolder = downloadFolder + deviceName
                core.log.message('Downloading result for device "'+deviceName+'"...')
                core.shell.mkdir(deviceFolder)
                artifactList = self.boto3.client.list_artifacts(arn=job['arn'], type='FILE')['artifacts']
                counter = {}
                for artifact in artifactList:
                    artifactName = artifact['name']
                    artifactUrl = artifact['url']
                    if not artifactName in counter: counter[artifactName] = 0
                    else: counter[artifactName] += 1
                    ext = '_' + str(counter[artifactName])
                    destination = core.shell.join_path(deviceFolder, artifactName+ext)
                    core.http.downloadFile(url=artifactUrl, destination=destination)
                    if artifactName == 'Customer Artifacts':
                        core.log.message('Handling customer artifact...')
                        core.shell.unzip(destination, deviceFolder)
                    if self.platform == core.const.Platform.iOS:
                        if artifactName == 'Application Output':
                            self._iosExtractXML(destination)
            core.log.message('All files downloaded')
            return True
        else:
            core.log.warning('Run "'+runArn+'" is not ready for download (Result='+str(result)+')')
            return False

    '''  TODO: Remove this workaround when customer artifact is available '''
    def _iosExtractXML(self, input):
        output = input + '.xml'
        inFile = open(input)
        outFile = open(output, 'w')
        buffer = []
        tokenStart = '[[PRINT_START]]'
        tokenEnd = '[[PRINT_END]]'
        record = 'NO'
        for line in inFile:
            if record == 'NO':
                index = line.find(tokenStart)
                if index != -1:
                    record = 'YES'
                    outFile.write(line[index + len(tokenStart):])
            elif record == 'YES':
                index = line.find(tokenEnd)
                if index != -1:
                    record = 'DONE'
                    outFile.write(line[0:index])
                    break
                else:
                    outFile.write(line)
        inFile.close()
        outFile.close()
        if record != 'DONE':
            core.shell.remove(output)
        return record == 'DONE'
