'''
Docker Adaptor
'''
import core
import docker
import time

class DockerAdaptor:
    dockerfilePath = 'scripts/automation/jobs/docker'
    
    ''' Constructor '''
    def __init__(self, config):
        core.log.message('Creating Docker Adaptor')
        self.client = docker.from_env()
        if not self.client:
            core.log.error('Unable to create docker client (daemon not started?)')
        core.log.message('Successfully created docker client')
        self._setupSSH(config)
   
    ''' Build docker image '''
    def buildImage(self, config):
        tag = 'auto:' + config.getPlatform().value.lower()
        dockerfile = config.getDockerFile()
        path = core.shell.os_path(config.getRootDir() + '/' + self.dockerfilePath)
        core.log.important('Creating image "' + tag + '" from "'+core.shell.os_path(path + '/' + dockerfile)+'"...')
        image, log = self.client.images.build(path=path, dockerfile=dockerfile, tag=tag, rm=True)
        if not image:
            core.log.error('Unable to create docker image')
        core.log.message('Successfully created docker image')
        return tag
    
    ''' Create and run container in specified image '''
    def runContainer(self, config, image, command=None):
        core.log.important('Running container from image "' + image + '"')
        mounts = [ docker.types.Mount(target='/root/.ssh/', source=self.sshDir, type='bind', read_only=True), docker.types.Mount(target='/build', source=config.getRootDir(), type='bind') ]
        container = self.client.containers.run(image=image, mounts=mounts, environment=config.getAutoEnv(), detach=True, command=command)
        core.log.message('Container ID: '+container.id)
        result = container.wait()
        logs = container.logs()
        container.remove()
        core.log.emit(logs, False)
        
        user = config.getUsername()
        group = user
        path = config.getRootDir()
        core.log.message('Changing ownership of "'+path+'" to "'+user+':'+group+'"...')
        core.shell.chown(path, user, group)

        exit = int(result['StatusCode'])
        return exit

    ''' Setup SSH for docker '''
    def _setupSSH(self, config):
        core.log.message('Preparing SSH client...')
        self.sshDir = core.shell.os_path(config.getRootDir() + '/.auto/.ssh')
        core.shell.mkdir(self.sshDir, deleteIfExists=True)
        core.shell.chmod(self.sshDir, 0o700)
        localSshPath = core.shell.join_path(core.shell.home_path(), '.ssh')
        dockerSsh = config.getDockerSsh()
        sshFiles = { dockerSsh: 'id_rsa', dockerSsh+'.pub': 'id_rsa.pub', 'known_hosts' : 'known_hosts' }
        for fileSrc, fileDst in sshFiles.items():
            srcPath = core.shell.join_path(localSshPath, fileSrc)
            dstPath = core.shell.join_path(self.sshDir, fileDst)
            core.shell.copy(srcPath, dstPath)
            if fileDst == 'known_hosts': core.shell.chmod(dstPath, 0o644)
            else: core.shell.chmod(dstPath, 0o600)
