'''
AWS S3 Adaptor
'''
import core
import os
import threading
import botocore
import mimetypes
from adaptor.awsboto3 import Boto3Adaptor

class S3Adaptor:
    ''' Constructor '''
    def __init__(self, config):
        core.log.message('Creating S3 adaptor')
        self.client = Boto3Adaptor(config, 's3').client
        if not self.client:
            core.log.error('Unable to obtain boto3 client for the S3 service')
        self.mime = mimetypes.MimeTypes()
            
    ''' Upload File '''
    def uploadFile(self, filename, bucket, object, extraArgs={}, config=None, exitOnError=True):
        try:
            extraArgs = dict(extraArgs)
            if 'ContentType' not in extraArgs.keys():
                contentType = self.mime.guess_type(filename)[0]
                if contentType: extraArgs['ContentType'] = contentType
            core.log.important('Uploading file "'+filename+'" to "s3://'+bucket+'/'+object+'" '+str(extraArgs)+'...')
            response = self.client.upload_file(filename, bucket, object, ExtraArgs=extraArgs, Config=config, Callback=ProgressCallback(filename, bucket+'/'+object, isUpload=True, verbose=False))
            return response
        except botocore.exceptions.ClientError as e:
            msg = 'AW3 S3 upload_file failed: '+str(e)
            if exitOnError: raise core.const.AutoException(msg, -1)
            else: core.log.warning(msg)
            return False

    ''' Upload directory '''
    def uploadDirectory(self, directory, bucket, object, extraArgs={}, config=None, exitOnError=True):
        offset = len(directory)
        core.log.important('Uploading directory "'+directory+'" to "s3://'+bucket+'/'+object+'"...')
        for root, _, files in os.walk(directory):
            for file in files:
                path = core.shell.os_path(root + '/' + file)
                target = object + path[offset:]
                self.uploadFile(path, bucket, target, extraArgs, config)
        return True
                
    ''' Download File '''
    def downloadFile(self, bucket, object, filename, extraArgs={}, config=None, exitOnError=True):
        try:
            response = self.client.download_file(bucket, object, filename, ExtraArgs=extraArgs, Config=config, Callback=ProgressCallback(filename, bucket+'/'+object, isUpload=False))
            return response
        except botocore.exceptions.ClientError as e:
            msg = 'AW3 S3 download_file failed: '+str(e)
            if exitOnError: raise core.const.AutoException(msg, -1)
            else: core.log.warning(msg)
            return None

    ''' Download directory '''
    def downloadDirectory(self, directory, bucket, object, extraArgs={}, config=None, exitOnError=True):
        # TODO
        pass

    ''' Create presigned URL '''
    def createPresignedUrl(self, bucket, object, expiration=3600, exitOnError=True):
        try:
            response = self.client.generate_presigned_url('get_object', Params={'Bucket': bucket, 'Key': object}, ExpiresIn=expiration)
            return response
        except botocore.exceptions.ClientError as e:
            msg = 'AW3 S3 generate_presigned_url failed: '+str(e)
            if exitOnError: raise core.const.AutoException(msg, -1)
            else: core.log.warning(msg)
            return None


''' Progress Callback '''
class ProgressCallback(object):
    ''' Constructor '''
    def __init__(self, src, dst, isUpload=True, verbose=True):
        self.src = src
        self.dst = dst
        self.verbose = verbose
        if isUpload:
            self.size = float(core.shell.file_size(src))
            self.dst = 's3://' + dst
        else:
            self.verbose = False
            self.size = 0 #TODO
            self.src = 's3://' + src
        self.seen_so_far = 0
        self.lock = threading.Lock()
            
    ''' Will be invoked by Boto3 '''
    def __call__(self, bytes_amount):
        with self.lock:
            self.seen_so_far += bytes_amount
            percentage = (self.seen_so_far / self.size) * 100
            if self.verbose:
                core.log.message('... %s ==> %s: %d/%d (%.2f%%)' % (self.src, self.dst, self.seen_so_far, self.size, percentage))
