'''
Environment and configuration module
'''
import os
import core.log
import core.const
import core.util
import version
import getpass
import socket

class Config:
    defaultJobTimeoutSec = 30 * 60
    
    ''' Constructor '''
    def __init__(self):
        # Base env
        self.rootDir = os.getcwd()
        self.frameworkDir = core.shell.script_path()
        self.hostname = socket.gethostname()
        self.username = getpass.getuser()
        self.env = os.environ.copy()
        self.displayEnv()
        
        # Job definition
        self.pipeline = self.parseString('AUTO_PIPELINE', 'default')
        self.platform = self.parseEnum({ core.const.Platform.Android, core.const.Platform.AndroidWeb, core.const.Platform.ChromeCast, core.const.Platform.iOS, core.const.Platform.iOSWeb, core.const.Platform.OSX, core.const.Platform.Posix, core.const.Platform.PS4, core.const.Platform.tvOS, core.const.Platform.VideoJS, core.const.Platform.Web, core.const.Platform.XboxOne, }, 'AUTO_PLATFORM')
        self.platformSdk = self.parseString('AUTO_PLATFORM_SDK', 'all')
        self.action = self.parseEnum({core.const.Action.Build,core.const.Action.Test, core.const.Action.Release,}, 'AUTO_ACTION')

        # Docker
        self.dockerRunning = self._isRunningInDocker()
        self.dockerSsh = self.parseString('AUTO_DOCKER_SSH', 'id_rsa')
        self.dockerInvoke = self.parseBool('AUTO_DOCKER_INVOKE', False)
        if self.dockerInvoke:
            if self.dockerRunning:
                core.log.warning('Docker invoke requested, but already running in container, disabling flag')
                self.dockerInvoke = False
            self.unsetEnvVar('AUTO_DOCKER_INVOKE')
        self.dockerFile = self.parseString('AUTO_DOCKER_FILE', 'Dockerfile')
        
        # Job base
        self.jobConfig = {}
        self.jobClean = self.parseBool('AUTO_JOB_CLEAN', True)
        self.jobSkip = self.parseBool('AUTO_JOB_SKIP', False)
        self.jobTimeoutSec = self.parseInt('AUTO_JOB_TIMEOUTSEC', self.defaultJobTimeoutSec)
        
        # Work/Output directories
        targetPath = self.platform.value.lower() + '/' + self.platformSdk.lower() + '/' + self.action.value.lower()
        self.workDir = core.shell.os_path(self.rootDir + '/.auto/work/' + targetPath)
        self.outDir = core.shell.os_path(self.rootDir + '/.auto/out/' + targetPath)
        core.log.message('Environment successfully parsed')

    ''' Whether we're running in a docker container '''
    def _isRunningInDocker(self):
        try:
            with open('/proc/1/cgroup', 'rt') as ifh:
                return 'docker' in ifh.read()
        except:
            return False

    ''' Check if given key in environment is a secret '''
    def isSecret(self, key):
        secretList = [
            'AUTO_AWS_SECRETKEY',
            'AUTO_GRID_SECRET',
            'AUTO_ARTIFACTORY_SECRET'
        ]
        return key in secretList
    
    ''' Read environment variable '''
    def readEnvVar(self, key, default = None):
        key = str(key)
        value = None
        if key in self.env.keys():
            value=self.env[key].strip()
        if not value and default:
            value = default
        return value

    ''' Write environment variable '''
    def writeEnvVar(self, key, value):
        self.env[str(key)] = str(value)
    
    ''' Unset environment variable '''
    def unsetEnvVar(self, key):
        self.env.pop(str(key), None)
    
    ''' Parse enum '''
    def parseEnum(self, map, key, default = None):
        envVal = self.readEnvVar(key)
        if envVal:
            envVal = envVal.lower()
            for enum in map:
                if enum.value.lower() == envVal:
                    return enum
            core.log.error('Invalid value for env.'+key+'('+envVal+'). Valid values are: '+str(map) + ')' )
        elif not default:
            core.log.error('Variable env.'+key+' not set')
        return default

    ''' Parse string '''
    def parseString(self, key, default = None):
        envVal = self.readEnvVar(key)
        if not envVal:
            if default == None:
                core.log.error('Value not specified for env.'+key)
            else:
                envVal = default
        return envVal

    ''' Parse integer '''
    def parseInt(self, key, default = None):
        return int(self.parseString(key, default))

    ''' Parse boolean '''
    def parseBool(self, key, default = None):
        envVal = self.readEnvVar(key)
        if not envVal:
            if default == False or default == True:
                return default
            else:
                core.log.error('Value not specified for env.'+key)

        envVal = envVal.lower()
        if envVal == 'true' or envVal == 'yes' or envVal == 'on':
            return True
        elif envVal == 'false' or envVal == 'no' or envVal == 'off':
            return False
        else:
            core.log.error('Invalid value specified for boolean key "'+key+'" (="'+envVal+'")')
    
    ''' Display environment '''
    def displayEnv(self):
        core.log.message( 'Automation Framework '+version.__version__+' ('+self.username+'@'+self.hostname+')', core.log.TextFormat.BOLD+core.log.TextFormat.BLUE)
        for k, v in sorted(self.env.items()):
            if k.startswith('AUTO_'):
                if self.isSecret(k): v = '*******'
                core.log.important( 'env.' + k + ': ' + v )

    ''' Return environment dictionary '''
    def getEnv(self):
        return self.env
    def getAutoEnv(self):
        envAuto = {}
        for k, v in self.env.items():
            if k.startswith('AUTO_'):
                envAuto[k] = v
        return envAuto
                
    ''' Return directory settings '''
    def getRootDir(self):
        return self.rootDir
    def getWorkDir(self):
        return self.workDir
    def getOutDir(self):
        return self.outDir

    ''' Return routing settings '''
    def getPipeline(self):
        return self.pipeline
    def getPlatform(self):
        return self.platform
    def getPlatformSdk(self):
        return self.platformSdk
    def getAction(self):
        return self.action
    
    ''' Return docker settings '''
    def getDockerInvoke(self):
        return self.dockerInvoke
    def getDockerSsh(self):
        return self.dockerSsh
    def getDockerRunning(self):
        return self.dockerRunning
    def getDockerFile(self):
        return self.dockerFile

    ''' Return host specific settings '''
    def getHostname(self):
        return self.hostname
    def getUsername(self):
        return self.username
    
    ''' Return job specific settings '''
    def getJobClean(self):
        return self.jobClean
    def getJobSkip(self):
        return self.jobSkip
    def getJobTimeoutSec(self):
        return self.jobTimeoutSec
    def getJobConfig(self):
        return self.jobConfig
    def jobSetting(self, key):
        if key in self.jobConfig:
            return self.jobConfig[key]
        else:
            core.log.error('The key "'+key+'" cannot be found in the job configuration')
