'''
    Logging module
'''
import core
import inspect
import sys

''' Output text format '''
class TextFormat:
    HEADER = '\033[95m'
    BLUE = '\033[94m'
    GREEN = '\033[92m'
    CYAN = '\033[36m'
    WARNING = '\033[33m'
    ERROR = '\033[91m'
    NORMAL = '\033[0m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'

''' Initialize Log '''
def init():
    try:
        # This will enable VT processing on Windows
        import ctypes
        import win32api
        ENABLE_PROCESSED_OUTPUT = 0x01
        ENABLE_WRAP_AT_EOL_OUTPUT = 0x02
        ENABLE_VIRTUAL_TERMINAL_PROCESSING = 0x04
        kernel32 = ctypes.windll.kernel32
        kernel32.SetConsoleMode(kernel32.GetStdHandle(win32api.STD_OUTPUT_HANDLE), 
            ENABLE_PROCESSED_OUTPUT|ENABLE_WRAP_AT_EOL_OUTPUT|ENABLE_VIRTUAL_TERMINAL_PROCESSING)
    except: pass
    
''' Emit and flush stdout with optional timestamp '''
def emit(what, timestamp=True):
    if timestamp: header = core.util.now() + ': '
    else: header = ''
    print(header + what)
    sys.stdout.flush()

''' Error output and optional auto-abort '''
def error(what, code=1, exit=True):
    emit(TextFormat.ERROR +'ERROR [' + str(code) +']: ' + str(what) + TextFormat.NORMAL)
    if exit:
        call_trace()
        raise core.const.AutoException(what, code)

''' Print message '''
def message(what, format=TextFormat.NORMAL):
    emit(format + str(what) + TextFormat.NORMAL)

''' Print important message '''
def important(what):
    message(what, TextFormat.BOLD)

''' Print system message '''
def system(what, failure=False):
    color = TextFormat.BOLD
    if not failure: color += TextFormat.BLUE
    else: color += TextFormat.ERROR
    message(what, color)

''' Print warning message '''
def warning(what):
    emit(TextFormat.WARNING + 'WARNING: ' + str(what) + TextFormat.NORMAL)

''' Print job prologue '''
def job_prologue(config):
    system( 'JOB STARTED: ' + str(config.getPipeline()) + ' ' + str(config.getPlatform()) + ' ' + str(config.getPlatformSdk()) + ' ' + str(config.getAction()))

''' Print job epilogue '''
def job_epilogue(config, exit = 0):
    if exit:
        failure = True
        status = 'FAILURE ['+str(exit)+']'
    else:
        failure = False
        status = 'SUCCESS'
    system( 'JOB '+status+': ' + str(config.getPipeline()) + ' ' + str(config.getPlatform()) + ' ' + str(config.getPlatformSdk()) + ' ' + str(config.getAction()), failure=failure)

''' Print call trace '''
def call_trace():
    message("Callstack:", TextFormat.BOLD + TextFormat.UNDERLINE)
    stack = inspect.stack()
    # ignore first and last two elements
    for i in range(2, len(stack)-3):
        record = stack[i]
        frame = record[0]
        info = inspect.getframeinfo(frame)
        message( '> ' + TextFormat.UNDERLINE + info.function + TextFormat.NORMAL + ' in ' + info.filename + '('+str(info.lineno)+')' )
