'''
Shell Utilities
'''
import subprocess
import sys
import stat
import core.log
import os
import shutil
import shlex
import fnmatch
import zipfile
import ntpath
import glob
import tempfile
import tarfile
import psutil

''' Textform for any shell operations '''
shellTextFormat = core.log.TextFormat.BLUE

''' Return home directory for current user '''
def home_path():
    return os.path.expanduser("~")

''' Return script path '''
def script_path():
    return os.path.dirname(os.path.realpath(sys.argv[0]))

''' Return file size '''
def file_size(file):
    return os.path.getsize(file)

''' Return file extension '''
def file_ext(file):
    return os.path.splitext(file)

''' Kill process by name '''
def kill(name, exactMatch=True):
    core.log.message('Killing all processes named "'+name+'" (exactMatch='+str(exactMatch)+')...')
    for proc in psutil.process_iter():
        procName = proc.name()
        kill = False
        if exactMatch:
            if name == procName: kill = True
        elif name.lower() in procName.lower(): kill = True
        if kill:
            proc.kill()
         
''' Tar/gz directory '''
def tar(src, dst):
    src = os_path(src)
    dst = os_path(dst)
    core.log.message('Creating tarball "'+src+'" => "'+dst+'"...', shellTextFormat)
    with tarfile.open(dst, 'w:gz') as handle:
        handle.add(src, arcname=os.path.basename(src))

''' untar/gz directory '''
def untar(src, dst='.'):
    src = os_path(src)
    dst = os_path(dst)
    core.log.message('Expanding tarball "'+src+'" => "'+dst+'"...', shellTextFormat)
    with tarfile.open(src, 'r:gz') as handle:
        handle.extractall(path=dst)

''' Zip file or directory '''
def zip(src, dst=None):
    if not dst:
        dst = src + '.zip'
    src = os_path(src)
    dst = os_path(dst)
    temp = tempfile.NamedTemporaryFile().name
    core.log.message('Zipping "'+src+'" => "'+dst+'"', shellTextFormat)
    root,base = path_split(src)
    pushd(root)
    zipf = zipfile.ZipFile(temp, 'w', zipfile.ZIP_DEFLATED)
    for root, dirs, files in os.walk(base):
        for file in files:
            core.log.message(' Adding \''+os.path.join(root, file)+'\'', shellTextFormat)
            zipf.write(os.path.join(root, file))
    zipf.close()
    popd()
    shutil.move(temp, dst)

''' Unzip archive '''
def unzip(src, dst):
    core.log.message('Unzipping "'+src+'" => "'+dst+'"', shellTextFormat)
    with zipfile.ZipFile(src, 'r') as zf:
        zf.extractall(dst)

''' Write file '''
def write_file(path, content, binary=False):
    mode = 'w'
    if binary: mode += 'b'
    path = os_path(path)
    core.log.message('Writing to file "' + path + '"...', shellTextFormat)
    with open(path, mode) as fp:
        fp.write(content)

''' Match objects by wildcard for a given path '''
def match(pattern, path):
    return glob.glob(join_path(path, pattern))

''' Find objects on the filesystem recursively '''
def find(pattern, path, findFiles=True, findDirs=True, errorOnNotFound=True):
    matches = []
    for root, dirnames, filenames in os.walk(os.path.abspath(path)):
        if findFiles:
            for filename in fnmatch.filter(filenames, pattern):
                matches.append(join_path(root, filename))
        if findDirs:
            for dirname in fnmatch.filter(dirnames, pattern):
                matches.append(join_path(root, dirname))
    if not matches and errorOnNotFound:
        core.log.error('find: "'+pattern+'" (files="'+str(findFiles)+'", dirs="'+str(findDirs)+'") not found in "'+path+'"')
    return matches

''' Sub Process Call '''
def _call_subprocess(cmd, exitOnError=True, contextDir=None, env=None, shell=False, checkOutput=False, verbose=True):
    if not cmd:
        core.log.warning('No command given')
        return 1
    if not env: env = os.environ.copy()
    if sys.platform.startswith('win'): shell = True
    if not shell: cmd = shlex.split(cmd)
    if verbose:
        core.log.message('Running command: "' + str(cmd)+'" with shell='+str(shell)+', contextDir="'+str(contextDir)+'"', shellTextFormat)
    status = True
    result = 0
    if contextDir:
        if not os.path.isdir(contextDir):
            core.log.message('Creating context directory "'+contextDir+'" before command execution', shellTextFormat)
            mkdir(contextDir)
        pushd(contextDir)
    try:
        if not checkOutput:
            result = subprocess.check_call(cmd, stderr=subprocess.STDOUT, shell=shell, env=env)
            if result: status = False
        else:
            result = subprocess.check_output(cmd, shell=shell, env=env)
    except subprocess.CalledProcessError as e:
        status = False
        result = e.returncode
    if contextDir: popd()
    if not status and exitOnError:
        if verbose: cmd = str(cmd)
        else: cmd = '<hidden>'
        core.log.error('Process invocation failed', result)
    return result

''' Execute external command '''
def run(cmd, exitOnError=True, contextDir=None, env=None, shell=False, verbose=True):
    return _call_subprocess(cmd=cmd, exitOnError=exitOnError, contextDir=contextDir, env=env, shell=shell, checkOutput=False, verbose=verbose)

''' Execute external command, capture result as binary string '''
def capture(cmd, exitOnError=True, contextDir=None, env=None, shell=False, verbose=True):
    return _call_subprocess(cmd=cmd, exitOnError=exitOnError, contextDir=contextDir, env=env, shell=shell, checkOutput=True, verbose=verbose)

''' Capture text output '''
def capture_text(cmd, exitOnError=True, contextDir=None, env=None, shell=False, verbose=True, trim=True):
    result = capture(cmd, exitOnError, contextDir, env, shell, verbose).decode()
    if trim: result = result.strip()
    return result

''' Change ownership '''
def chown(path, user, group):
    uid = -1
    gid = -1
    try:
        import pwd
        import grp
        uid = pwd.getpwnam(user).pw_uid
        gid = grp.getgrnam(group).gr_gid
    except: pass
    if uid != -1:
        if gid != -1: spec = user+':'+group
        else: spec = user
        core.shell.run(cmd='sudo chown -R '+spec+' '+path, shell=True)
        '''
        TODO: This fails on Linux
        for root, dirs, files in os.walk(path):
            for dir in dirs:
                path = os.path.join(root, dir)
                os.chown(path, uid, gid)
            for file in files:
                path = os.path.join(root, file)
                os.chown(path, uid, gid)
        '''
        
''' Move file or directory '''
def move(src, dst):
    src = os_path(src)
    dst = os_path(dst)
    shutil.move(src, dst)

''' Copy directory (recursively) '''
def copytree(src, dst, deleteIfExists=True):
    src = os_path(src)
    dst = os_path(dst)
    core.log.message('Copying directory "'+src+'" to "'+dst+'"...', shellTextFormat)
    if deleteIfExists and os.path.isdir(dst):
        rmdir(dst)
    shutil.copytree(src, dst)

''' Copy file '''
def copyfile(src, dst):
    src = os_path(src)
    dst = os_path(dst)
    core.log.message('Copying file "'+src+'" to "'+dst+'"...', shellTextFormat)
    path, file = path_split(dst)
    if path: mkdir(path)
    shutil.copy(src, dst)

''' Copy object (file or directory) '''
def copy(src, dst):
    src = os_path(src)
    dst = os_path(dst)
    if os.path.isdir(src): copytree(src, dst)
    else: copyfile(src, dst)

''' Rename file or directory '''
def rename(src, dst):
    src = os_path(src)
    dst = os_path(dst)
    try:
        os.rename(src, dst)
    except:
        core.log.error('Rename '+src+' to '+dst+' failed')

''' Remove file or directory '''
def remove(path):
    path = os_path(path)
    if os.path.exists(path):
        if os.path.isdir(path):
            rmdir(path)
        else:
            os.remove(path)

''' Change permissions '''
def chmod(path, flags):
    path = os_path(path)
    os.chmod(path, flags)

''' Implements rmtree (works on Windows too) '''     
def rmdir(top):
    top = os_path(top)
    for root, dirs, files in os.walk(top, topdown=False):
        for name in files:
            filename = os.path.join(root, name)
            if not os.path.islink(filename): os.chmod(filename, stat.S_IWUSR)
            os.remove(filename)
        for name in dirs:
            os.rmdir(os.path.join(root, name))
    os.rmdir(top)    
    
''' Make directory '''
def mkdir(dir, deleteIfExists=False, mode=0o755):
    if not dir:
        core.log.error('Invalid directory specified')
    dir = os_path(dir)
    if os.path.isdir(dir):
        if deleteIfExists:
            core.log.message('Deleting existing directory "' + dir + '" before recreating...', shellTextFormat)
            rmdir(dir)
            os.makedirs(dir, mode)
    else:
        os.makedirs(dir, mode)

''' Make directories '''
def mkdirs(dirs, deleteIfExists=False, mode=0o755):
    for dir in dirs:
        mkdir(dir, deleteIfExists=deleteIfExists, mode=mode)

''' Return base of path '''
def basename(path):
    path = os_path(path)
    head, tail = ntpath.split(path)
    return tail or ntpath.basename(head)

''' Split path into directory and file '''
def path_split(path_and_file):
    path_and_file = os_path(path_and_file)
    return os.path.split(path_and_file)

''' Split file into name and extension '''
def filename_split(name_and_ext):
    return os.path.splitext(name_and_ext)

''' Concatenate two paths '''
def join_path(lhs, rhs):
    lhs = os_path(lhs)
    rhs = os_path(rhs)
    return os.path.join(lhs, rhs)

''' Convert paths to OS specific '''
def os_path(path):
    return path.replace('/', os.sep)

''' Push directory '''
pushd_stack = []
def pushd(dir):
    dir = os_path(dir)
    pushd_stack.append( os.getcwd() )
    core.log.message( 'pushd: "' + os.getcwd() + '" -> "' + dir + '"', shellTextFormat )
    try:
        os.chdir(dir)
    except:
        core.log.error('os.chdir to "'+dir+'" failed')

''' Pop directory '''
def popd():
    if pushd_stack:
        dir = pushd_stack.pop();
        core.log.message( 'popd: "' + os.getcwd() + '" -> "' + dir + '"', shellTextFormat )
        try:
            os.chdir(dir)
        except:
            core.log.error('os.chdir to "'+dir+'" failed')
