'''
Common Job Configurations
'''
import core
from adaptor.awsboto3 import Boto3Adaptor
from adaptor.s3 import S3Adaptor
from adaptor.browsergrid import BrowserGridAdaptor
from adaptor.selenium import SeleniumAdaptor
from adaptor.artifactory import ArtifactoryAdaptor
from enum import Enum

''' Enumerates test hosts '''
class TestHost(Enum):
    DeviceFarm = 'Devicefarm'
    Emulator = 'Emulator'
    PhysicalDevice = 'PhysicalDevice'
    
''' Enumerates E2E test modes '''
class E2ETestMethod(Enum):
    BrowserGrid = 'grid'
    ChromeDevTools = 'chrome-devtools'
    Local = 'local'

''' Build Configuration Dispatcher '''
def configureBuild(config):
    jobConfig = config.getJobConfig()
    jobConfig['build.cmakeonly'] = config.parseBool('AUTO_BUILD_CMAKEONLY', False)
    return config

''' Release Configuration Dispatcher '''
def configureRelease(config, restrict={core.const.ReleaseEnv.Dev, core.const.ReleaseEnv.Test, core.const.ReleaseEnv.Prod}, default=None):
    jobConfig = config.getJobConfig()
    jobConfig['release.env'] = config.parseEnum(restrict, 'AUTO_RELEASE_ENV', default)
    jobConfig['release.branch'] = config.parseString('AUTO_VCS_BRANCH')
    jobConfig['release.commit'] = config.parseString('AUTO_VCS_COMMIT', '') # Optional
    jobConfig['release.artifact'] = core.shell.os_path(config.getRootDir() + '/.auto/out/' + config.getPlatform().value.lower() + '/' + config.getPlatformSdk().lower() + '/build')
    target = config.parseEnum({core.const.ReleaseTarget.S3, core.const.ReleaseTarget.Artifactory,}, 'AUTO_RELEASE_TARGET')
    jobConfig['release.target'] = target
    core.log.message('Configure release testing for ' + jobConfig['release.target'].value+'/'+jobConfig['release.env'].value)
    if target == core.const.ReleaseTarget.S3:
        configureS3Release(config)
    elif target == core.const.ReleaseTarget.Artifactory:
        configureArtifactoryRelease(config)
    core.log.message('Release Configuration: '+str(config.getJobConfig()))
    return config

''' Configure S3 Release '''
def configureS3Release(config):
    jobConfig = config.getJobConfig()
    jobConfig['aws.s3'] = S3Adaptor(config)
    if jobConfig['release.env'] == core.const.ReleaseEnv.Dev:
        bucket = 'cvp-web.twitch.tv'
    elif jobConfig['release.env'] == core.const.ReleaseEnv.Test:
        bucket = 'cvp-web-test.twitch.tv'
    elif jobConfig['release.env'] == core.const.ReleaseEnv.Prod:
        bucket = 'cvp-web-prod.twitch.tv'
    jobConfig['release.s3.bucket'] = bucket
    return config

''' Configure Artifactory Release '''
def configureArtifactoryRelease(config):
    jobConfig = config.getJobConfig()
    host = config.parseString('AUTO_ARTIFACTORY_HOST')
    user = config.parseString('AUTO_ARTIFACTORY_USER', 'SRVC-VideoPlayback-Automation')
    secret = config.parseString('AUTO_ARTIFACTORY_SECRET')
    registry = config.parseString('AUTO_ARTIFACTORY_REGISTRY', 'https://npm.internal.justin.tv')
    jobConfig['release.artifactory'] = ArtifactoryAdaptor(config, host, user, secret, registry)
    return config

''' Test Configuration Dispatcher '''
def configureTesting(config, restrict={core.const.TestType.Unit, core.const.TestType.Integration, core.const.TestType.E2E}, default=None):
    jobConfig = config.getJobConfig()
    type = config.parseEnum(restrict, 'AUTO_TEST_TYPE', default)
    jobConfig['test.type'] = type
    if type == core.const.TestType.Unit:
        configureUnitTesting(config)
    elif type == core.const.TestType.Integration:
        configureIntegrationTesting(config)
    elif type == core.const.TestType.E2E:
        configureE2ETesting(config)
    elif type == core.const.TestType.Custom:
        configureCustomTesting(config)
    else:
        core.log.error("Unknown test type: "+str(type))
    core.log.message('Testing Configuration: '+str(config.getJobConfig()))
    return config

''' Configure Unit testing type '''
def configureUnitTesting(config):
    jobConfig = config.getJobConfig()
    jobConfig['test.buildtype'] = config.parseEnum({core.const.BuildType.Debug, core.const.BuildType.Release,}, 'AUTO_BUILD_TYPE', core.const.BuildType.Debug).value
    return config

''' Configure Integration testing type '''
def configureIntegrationTesting(config):
    jobConfig = config.getJobConfig()
    jobConfig['test.host'] = config.parseEnum({TestHost.Emulator, TestHost.DeviceFarm, TestHost.PhysicalDevice}, 'AUTO_TEST_HOST')
    if jobConfig['test.host'] == TestHost.PhysicalDevice: jobConfig['test.deviceip'] = config.parseString('AUTO_TEST_DEVICEIP')
    jobConfig['test.buildtype'] = config.parseEnum({core.const.BuildType.Debug, core.const.BuildType.Release,}, 'AUTO_BUILD_TYPE', core.const.BuildType.Debug).value
    jobConfig['test.liveurl'] = config.parseString('AUTO_TEST_LIVEURL', 'https://www.twitch.tv/bufferbunny')
    jobConfig['test.vodurl'] = config.parseString('AUTO_TEST_VODURL', 'https://www.twitch.tv/videos/306892197')
    jobConfig['test.filter'] = config.parseString('AUTO_TEST_FILTER', '*')
    return config

''' Configure E2E testing type '''
def configureE2ETesting(config):
    jobConfig = config.getJobConfig()
    jobConfig['test.branch'] = config.parseString('AUTO_VCS_BRANCH')
    jobConfig['test.host'] = config.parseString('AUTO_TEST_HOST', 'https://cvp-web.internal.justin.tv/'+jobConfig['test.branch']+'/test.html')
    jobConfig['test.liveurl'] = config.parseString('AUTO_TEST_LIVEURL', 'https://www.twitch.tv/bufferbunny')
    jobConfig['test.vodurl'] = config.parseString('AUTO_TEST_VODURL', 'https://www.twitch.tv/videos/306892197')
    jobConfig['test.method'] = config.parseEnum({E2ETestMethod.BrowserGrid, E2ETestMethod.ChromeDevTools,E2ETestMethod.Local}, 'AUTO_E2E_METHOD')
    selMaxInstances = config.parseInt('AUTO_SELENIUM_MAXINSTANCES', 4)
    jobConfig['test.selenium'] = SeleniumAdaptor(config, selMaxInstances)
    config.writeEnvVar('PLAYER_BASEURL', jobConfig['test.host'])
    config.writeEnvVar('PLAYER_LIVEURL', jobConfig['test.liveurl'])
    config.writeEnvVar('PLAYER_VODURL', jobConfig['test.vodurl'])
    if jobConfig['test.method']==E2ETestMethod.BrowserGrid:
        gridHost = config.parseString('AUTO_GRID_HOST', 'cbg.internal.justin.tv')
        gridUser = config.parseString('AUTO_GRID_USER', 'player-core')
        gridSecret = config.parseString('AUTO_GRID_SECRET')
        jobConfig['test.browsergrid'] = BrowserGridAdaptor(config, gridHost, gridUser, gridSecret)
    return config

''' Configure Custom testing '''
def configureCustomTesting(config):
    jobConfig = config.getJobConfig()
    jobConfig['test.version'] = config.parseString('AUTO_TEST_VERSION', 'master')
    jobConfig['test.manifesturl'] = config.parseString('AUTO_TEST_MANIFESTURL', 'https://www.twitch.tv/bufferbunny')
    return config

''' Configure Devicefarm testing '''
def configureDevicefarmTesting(config):
    jobConfig = config.getJobConfig()
    jobConfig['aws.boto3'] = Boto3Adaptor(config, 'devicefarm')
    jobConfig['devicefarm.project'] = config.parseString('AUTO_DEVICEFARM_PROJECT')
    jobConfig['devicefarm.devicePool'] = config.parseString('AUTO_DEVICEFARM_DEVICEPOOL')
    return config

