'''
-----------------------------------------------
iOS/tvOS Build Job
Generates:
    - Debug and Release framework
    - SamplePlayer IPA
    - Doxygen documentation
-----------------------------------------------
Parameters:
    - AUTO_PLATFORM=iOS|tvOS
    - AUTO_ACTION=build
-----------------------------------------------
'''
import core
import jobs.config
from adaptor.cmake import CMakeAdaptor
from adaptor.xcode import XCodeAdaptor, lipo_create, lipo_info, export_archive
import re

class Runner:
    ''' Execute job '''
    def execute(self, config):
        self.buildFramework(config, 'Debug')
        self.buildFramework(config, 'Release')
        self.buildSampleIPA(config, 'Release')
        self.buildDocs(config)
        return 0
    
    def buildFramework(self, config, buildType):
        self._configure(config, buildType)
        core.log.important('Building framework for the "' + self.buildType+'" configuration...')
                           
        # Build for physical device
        cmakeAdaptor = CMakeAdaptor(generator='Xcode', root=config.getRootDir(), buildType=self.buildType, toolChain=self.toolChainPhysical, params='-DBUILD_EXAMPLES=ON -DENABLE_CODE_COVERAGE=OFF -DBUILD_TESTS=OFF')
        cmakeAdaptor.generate(contextDir=self.workDirPhysical)
        xcodeAdaptor = XCodeAdaptor(actions='build', configuration=self.buildType, project='player-core.xcodeproj', destination=self.destination, scheme='PlayerCore', options='ENABLE_BITCODE=NO ONLY_ACTIVE_ARCH=NO -parallelizeTargets -jobs 8')
        xcodeAdaptor.xcodebuild(contextDir=self.workDirPhysical)
        
        # Build for Simulator...
        cmakeAdaptor.generate(toolChain=self.toolChainSimulator, contextDir=self.workDirSimulator)
        xcodeAdaptor.xcodebuild(contextDir=self.workDirSimulator)
        
        # Combine builds to fat framework (physical + simulator binaries)
        product='PlayerCore'
        framework=product + '.framework'
        pathPhysical = core.shell.os_path(config.getWorkDir() + '/' + self.buildType + '/' + self.platformPhysical + '/platforms/ios/' + self.buildType+'-' + self.osPhysical + '/'+framework)
        pathSimulator = core.shell.os_path(config.getWorkDir() + '/' + self.buildType + '/' + self.platformSimulator + '/platforms/ios/' + self.buildType+'-' + self.osSimulator +'/'+framework)
        frameworkOutDir = core.shell.os_path(config.getOutDir() + '/' + framework)
        core.shell.copy(pathPhysical, frameworkOutDir)
        lipoOut = frameworkOutDir+'/'+product
        lipo_create(pathPhysical+'/'+product, pathSimulator+'/'+product, lipoOut)
        lipo_info(lipoOut)
        
        # Add podspec file and parse version
        podSpecPath=core.shell.os_path(config.getRootDir() + '/platforms/ios/PlayerCore.podspec')
        core.shell.copy(podSpecPath, frameworkOutDir)
        result = core.parser.key_value(podSpecPath, ['spec.version'])
        if not result: core.log.error('Unable to retrieve version from POD spec file')
        version = result[0][1].strip('\"')
        core.log.important('Parsed POD version: '+version)
        with open(core.shell.os_path(config.getOutDir() + '/version'), 'w') as handle:
            handle.write(version)
        
        # Create tarball and clean up
        core.shell.tar(frameworkOutDir, config.getOutDir()+'/PlayerCore_'+self.buildType+'_'+version+'.tgz')
        core.shell.remove(frameworkOutDir)
        return 0
    
    def buildSampleIPA(self, config, buildType):
        self._configure(config, buildType)
        core.log.important('Building Sample Player IPA for the "' + self.buildType+'" configuration...')
                                              
        archivePath = 'SamplePlayer.xcarchive'
        ipaPath = core.shell.os_path(config.getOutDir()+'/Sample')
        cmakeAdaptor = CMakeAdaptor(generator='Xcode', root=config.getRootDir(), buildType=self.buildType, toolChain=self.toolChainPhysical, params='-DBUILD_EXAMPLES=ON -DENABLE_CODE_COVERAGE=OFF -DBUILD_TESTS=OFF');
        cmakeAdaptor.generate(contextDir=self.workDirPhysical)
        actions = 'build archive'
        xcodeAdaptor = XCodeAdaptor(actions=actions, configuration=self.buildType, project='player-core.xcodeproj', destination=self.destination, scheme='SamplePlayer', options='-archivePath '+archivePath+' ENABLE_BITCODE=NO ONLY_ACTIVE_ARCH=NO -parallelizeTargets -jobs 8 DEVELOPMENT_TEAM=94324UXTV2 PRODUCT_BUNDLE_IDENTIFIER="tv.twitch.player.sample"')
        xcodeAdaptor.xcodebuild(contextDir=self.workDirPhysical)
        export_archive(archivePath=archivePath, exportPath=ipaPath, exportOptionsPlist=self.exportOptionsPlist, contextDir=self.workDirPhysical)
        core.shell.copy(ipaPath + '/SamplePlayer.ipa', config.getOutDir())
        core.shell.remove(ipaPath)
        return 0

    def buildDocs(self, config):
        core.log.important('Building DoxyGen docs...')
        workDir = core.shell.os_path(config.getWorkDir() + '/'+self.buildType + '/'+self.platformPhysical)
        core.shell.run(cmd='doxygen', contextDir=workDir)
        core.shell.tar(workDir+'/docs/html/', config.getOutDir()+'/docs.tgz')
        return 0
    
    ''' Configure Job '''
    def _configure(self, config, buildType):
        self.buildType = buildType
        if config.getPlatform() == core.const.Platform.iOS:
            self.platformPhysical = 'iPhone'
            self.osPhysical = 'iphoneos'
            self.toolChainPhysical = core.shell.os_path(config.getRootDir() + '/cmake/toolchain/'+self.platformPhysical+'.cmake')
            self.workDirPhysical = core.shell.os_path(config.getWorkDir() + '/' + self.buildType + '/' + self.platformPhysical)
            self.platformSimulator = 'iPhoneSimulator'
            self.osSimulator = 'iphonesimulator'
            self.toolChainSimulator = core.shell.os_path(config.getRootDir() + '/cmake/toolchain/'+self.platformSimulator + '.cmake')
            self.workDirSimulator = core.shell.os_path(config.getWorkDir() + '/' + self.buildType + '/' + self.platformSimulator)
            self.destination = 'generic/platform=iOS'
            self.exportOptionsPlist = core.shell.os_path(config.getRootDir()+'/examples/ios/export/SamplePlayer.plist')
        elif config.getPlatform() == core.const.Platform.tvOS:
            self.platformPhysical = 'AppleTV'
            self.osPhysical = 'appletvos'
            self.toolChainPhysical = core.shell.os_path(config.getRootDir() + '/cmake/toolchain/'+self.platformPhysical+'.cmake')
            self.workDirPhysical = core.shell.os_path(config.getWorkDir() + '/' + self.buildType + '/' + self.platformPhysical)
            self.platformSimulator = 'AppleTVSimulator'
            self.osSimulator = 'appletvsimulator'
            self.toolChainSimulator = core.shell.os_path(config.getRootDir() + '/cmake/toolchain/'+self.platformSimulator + '.cmake')
            self.workDirSimulator = core.shell.os_path(config.getWorkDir() + '/' + self.buildType + '/' + self.platformSimulator)
            self.destination = 'generic/platform=tvOS'
            self.exportOptionsPlist = core.shell.os_path(config.getRootDir()+'/examples/ios/export/SamplePlayerTV.plist')
        else: core.log.error('Invalid platform for this job')
