'''
iOS Test Job
---------------------------------------------------------------------------------------------
Supports testing in the emulator and on devicefarm
---------------------------------------------------------------------------------------------
AUTO_TEST_HOST: Specifies the host ('Devicefarm' or 'Emulator')

Devicefarm:
AUTO_AWS_REGION: The AWS region
AUTO_AWS_ACCESSKEY: Public Access Key
AUTO_AWS_SECRETKEY: Private Access Key
AUTO_DEVICEFARM_PROJECT: Devicefarm Project
AUTO_DEVICEFARM_DEVICEPOOL: Devicefarm Device Pool

Parameters: (Optional)
AUTO_BUILD_TYPE: Build type of the test application
AUTO_TEST_LIVEURL: The URL of the LIVE content to test
AUTO_TEST_VODURL: The URL of the VOD content to test
AUTO_TEST_FILTER: C++ Test framework filter spec (e.g. *, or MyTestSuite.MyTestCase)
'''
import core
import jobs.config
from adaptor.devicefarm import DeviceFarmAdaptor
from adaptor.cmake import CMakeAdaptor
from adaptor.xcode import XCodeAdaptor

class Runner:
    ''' Execute job '''
    def execute(self, config):
        jobs.config.configureTesting(config, restrict={core.const.TestType.Integration, core.const.TestType.Custom}, default=core.const.TestType.Integration)
        testType = config.jobSetting('test.type')
        if testType==core.const.TestType.Integration:
            return self.integrationTesting(config)
        elif testType==core.const.TestType.Custom:
            return self.customTesting(config)
        return 1

    ''' Run integration testing '''
    def integrationTesting(self, config):
        self.build(config)
        host = config.jobSetting('test.host')
        if host == jobs.config.TestHost.Emulator:
            return self.testOnEmulator(config)
        elif host == jobs.config.TestHost.DeviceFarm:
            return self.testOnDeviceFarm(jobs.config.configureDevicefarmTesting(config))
        return 1

    ''' Run custom testing '''
    def customTesting(self, config):
        # TODO: Download pre-compiled test package from artifactory/s3
        
        # HACK-START
        jobConfig = config.getJobConfig()
        jobConfig['test.liveurl'] = jobConfig['test.manifesturl']
        jobConfig['test.filter'] = 'PlayerMetadataTest.ValidateID3'
        jobConfig['test.host'] = jobs.config.TestHost.DeviceFarm
        jobConfig['test.buildtype'] = core.const.BuildType.Debug.value
        self.build(config)
        # HACK-END
        
        return self.testOnDeviceFarm(jobs.config.configureDevicefarmTesting(config))
    
    ''' Build test app '''
    def build(self, config):
        host = config.jobSetting('test.host')
        if host == jobs.config.TestHost.Emulator:
            toolChainName='iPhoneSimulator.cmake'
        elif host == jobs.config.TestHost.DeviceFarm:
            toolChainName='iPhone.cmake'
        buildType = config.jobSetting('test.buildtype')
        core.log.message('Running build for the "'+buildType+'" configuration...')
        
        # Run cmake
        cmakeAdaptor = CMakeAdaptor(generator='Xcode', root=config.getRootDir(), buildType=buildType, toolChain=core.shell.os_path(config.getRootDir() + '/cmake/toolchain/' + toolChainName), params='-DBUILD_EXAMPLES=OFF -DENABLE_CODE_COVERAGE=OFF -DBUILD_TESTS=ON', contextDir=config.getWorkDir());
        cmakeAdaptor.generate()
        
        # Run xcodebuild
        xcodeAdaptor = XCodeAdaptor(actions='build-for-testing', configuration=buildType, project='player-core.xcodeproj', destination='generic/platform=iOS', scheme='TestDriver', options="DEVELOPMENT_TEAM=94324UXTV2 CODE_SIGN_STYLE=Manual PROVISIONING_PROFILE_SPECIFIER=\"PlayerCore Sample Development\" -parallelizeTargets -jobs 8 GCC_PREPROCESSOR_DEFINITIONS='$GCC_PREPROCESSOR_DEFINITIONS AUTO_TESTURL=\\\""+config.jobSetting('test.liveurl') + "\\\" AUTO_TESTFILTER=\""+config.jobSetting('test.filter')+"\"'")
        xcodeAdaptor.xcodebuild(contextDir=config.getWorkDir())
        return 0
            
    ''' Run test on emulator '''
    def testOnEmulator(self, config):
        core.log.message('Testing on emulator...')
        # Setup testdriver session directory
        sessionDir = '/tmp/TestDriver'
        core.shell.mkdir(sessionDir, deleteIfExists=True)
        
        # Run xcodebuild
        xcodeAdaptor = XCodeAdaptor(actions='test-without-building', configuration=config.jobSetting('test.buildtype'), project='player-core.xcodeproj', destination='platform=\'iOS Simulator\',name=\'iPhone X\'', scheme='TestDriver')
        result = xcodeAdaptor.xcodebuild(contextDir=config.getWorkDir(), exitOnError=False)
       
        # Publish results
        core.util.publishTestResults(sessionDir, core.shell.join_path(config.getOutDir(), 'results'))
        return 0
    
    ''' Run tests on AWS devicefarm '''
    def testOnDeviceFarm(self, config):
        core.log.message('Testing on devicefarm...')
        devicefarmAdaptor = DeviceFarmAdaptor(config, core.const.Platform.iOS)
        devicefarmAdaptor.execute(config)
        return 0

