'''
    tvOS Test Job
    ---------------------------------------------------------------------------------------------
    Supports testing in the emulator
    ---------------------------------------------------------------------------------------------
    AUTO_BUILD_TYPE: (Optional) Set the build type of the test application and SDK ('Debug' or 'Release'). The default is 'Debug'
    AUTO_EMULATORNAME: (Optional) Set the emulator name. The default is 'Apple TV'
    AUTO_TEST_HOST: Specifies the host (i.e. 'Emulator')
    AUTO_TEST_LIVEURL: The URL to test against (i.e. https://www.twitch.tv/bufferbunny)
'''
import core
import jobs.config
from adaptor.cmake import CMakeAdaptor
from adaptor.xcode import XCodeAdaptor

class Runner:
    ''' Execute job '''
    def execute(self, config):
        # Obtain jobs.specific test configuration
        jobs.config.configureTesting(config, restrict={core.const.TestType.Integration}, default=core.const.TestType.Integration)
        # Build APP
        self.build(config)
        # Run requested test method
        host = config.jobSetting('test.host')
        if host == jobs.config.TestHost.Emulator:
            return self.testOnEmulator(config)
        return 0

    ''' Build test app '''
    def build(self, config):
        host = config.jobSetting('test.host')
        if host == jobs.config.TestHost.Emulator:
            toolChainName='AppleTVSimulator.cmake'
        else:
            core.log.error('Invalid testhost ("+str(host)+") specified for this job')
        
        buildType = config.jobSetting('test.buildtype')
        core.log.message('Running build for the "'+buildType+'" configuration...')
    
        # Run cmake
        cmakeAdaptor = CMakeAdaptor(generator='Xcode', root=config.getRootDir(), buildType=buildType, toolChain=core.shell.os_path(config.getRootDir() + "/cmake/toolchain/" + toolChainName), params='-DBUILD_EXAMPLES=OFF -DENABLE_CODE_COVERAGE=OFF -DBUILD_TESTS=ON', contextDir=config.getWorkDir());
        cmakeAdaptor.generate()

        # Run xcodebuild
        xcodeAdaptor = XCodeAdaptor(actions='build-for-testing', configuration=buildType, project='player-core.xcodeproj', destination='generic/platform=tvOS', scheme='TestDriver', options="DEVELOPMENT_TEAM=94324UXTV2 CODE_SIGN_STYLE=Manual PROVISIONING_PROFILE_SPECIFIER=\"PlayerCore Sample Development\" -parallelizeTargets -jobs 8 GCC_PREPROCESSOR_DEFINITIONS='$GCC_PREPROCESSOR_DEFINITIONS AUTO_TESTURL=\\\""+config.jobSetting('test.liveurl') + "\\\" AUTO_TESTFILTER=\""+config.jobSetting('test.filter')+"\"'")
        xcodeAdaptor.xcodebuild(contextDir=config.getWorkDir())
    
    ''' Run test on emulator '''
    def testOnEmulator(self, config):
        core.log.message('Testing on emulator...')
        # Setup testdriver session directory
        sessionDir = '/tmp/TestDriver'
        core.shell.mkdir(sessionDir, deleteIfExists=True)
        
        # Run xcodebuild
        xcodeAdaptor = XCodeAdaptor(actions='test-without-building', configuration=config.jobSetting('test.buildtype'), project='player-core.xcodeproj', destination='platform=\'tvOS Simulator\',name=\'Apple TV\'', scheme='TestDriver')
        result = xcodeAdaptor.xcodebuild(contextDir=config.getWorkDir(), exitOnError=False)
        
        # Publish results
        core.util.publishTestResults(sessionDir, core.shell.join_path(config.getOutDir(), 'results'))
        return 0

