#include "BufferFilter.hpp"

namespace twitch {
namespace abr {
std::string BufferFilter::Name = "BufferFilter";
const MediaTime UpSwitchMinBuffer(5.0f);

enum class SwitchDirection {
    Up,
    Down,
    None
};

BufferFilter::BufferFilter(Log& log)
    : m_log(log)
    , m_upSwitchMinBuffer(UpSwitchMinBuffer)
    , m_lowLatencyMode(false)
{
}

bool BufferFilter::filter(const std::vector<Quality>& qualities, Filter::Context& context)
{
    const Quality& selected = context.getSelected();
    Quality target = context.getTarget(qualities);

    // use playback rate to determine effective buffer duration
    MediaTime buffered = context.getBufferDuration() / context.getPlaybackRate();
    SwitchDirection switchDirection = SwitchDirection::None;

    if (target.bitrate != selected.bitrate) {
        // if moving up must have at least the min buffer threshold
        // if moving down buffer must be less than the max buffer threshold
        if (target.bitrate > selected.bitrate && buffered >= m_upSwitchMinBuffer) {
            switchDirection = SwitchDirection::Up;
        } else if (target.bitrate < selected.bitrate) {
            switchDirection = SwitchDirection::Down;
        }
    }

    if (selected.bitrate == 0 || switchDirection != SwitchDirection::None) {
        m_log.info("select %s (%d) buffered %.2f s",
            selected.name.c_str(), selected.bitrate, buffered.seconds());
    } else {
        if (target.bitrate != selected.bitrate) {
            m_log.info("can't switch to %s (%d) buffered %.2f s target %.2f",
                target.name.c_str(), target.bitrate,
                buffered.seconds(),
                m_upSwitchMinBuffer.seconds());
            target = selected;
        }
    }

    // filter all but the target
    for (const auto& quality : qualities) {
        if (quality != target) {
            context.filter(*this, quality);
        }
    }

    return true;
}

void BufferFilter::onStatistics(const Statistics& statistics, const Quality& quality)
{
    (void)statistics;
    (void)quality;
}

void BufferFilter::onStreamChange()
{
}

void BufferFilter::setTargetBufferSize(MediaTime duration)
{
    if (duration > MediaTime::zero()) {
        // allowable gap since it's unlikely the buffer can be fully filled to this level
        const MediaTime threshold(1.0);
        MediaTime min = duration - std::min(duration, threshold);
        m_upSwitchMinBuffer = std::min(min, UpSwitchMinBuffer);
    }
}
}
}
