#include "DroppedFrameFilter.hpp"

namespace twitch {
namespace abr {
std::string DroppedFrameFilter::Name = "DroppedFrameFilter";
const int IntervalPeriod = 10;

DroppedFrameFilter::DroppedFrameFilter(Log& log)
    : m_log(log)
    , m_lastDecodedFrames(0)
    , m_lastDroppedFrames(0)
    , m_droppedFrameAverage(MediaTime(1.0) * IntervalPeriod)
{
}

bool DroppedFrameFilter::filter(const std::vector<Quality>& qualities, Filter::Context& context)
{
    (void)qualities;
    for (const auto& f : m_dropped) {
        context.filter(*this, f);
    }
    return true;
}

void DroppedFrameFilter::onStatistics(const Statistics& statistics, const Quality& quality)
{
    // compute the average dropped frames over the last 10 seconds and switch the quality down
    // if its too high
    MediaTime now = MediaTime::now();
    MediaTime elapsed = now - m_lastUpdate;

    // reset the count if the previous is bigger
    m_lastDroppedFrames = std::min(m_lastDroppedFrames, statistics.getDroppedFrames());
    m_lastDecodedFrames = std::min(m_lastDecodedFrames, statistics.getDecodedFrames());

    int dropped = statistics.getDroppedFrames() - m_lastDroppedFrames;
    int decoded = statistics.getDecodedFrames() - m_lastDecodedFrames;

    // ignore a large number of dropped frames (e.g. on web browsers when tab is out of focus)
    if (dropped > decoded) {
        if (decoded) { // can be negative
            m_log.warn("Ignore dropped %d decoded %d", dropped, decoded);
        }
        return;
    }

    m_droppedFrameAverage.add(dropped);

    if (m_droppedFrameAverage.elapsed(elapsed)) {
        int averageDroppedFrames = m_droppedFrameAverage.average();

        const int DroppedFramesThreshold = 5 * IntervalPeriod;
        const int MinFramerate = 30;
        if (averageDroppedFrames > DroppedFramesThreshold && quality.framerate > MinFramerate) {
            m_log.warn("Excessive frames dropped average %d total %d",
                averageDroppedFrames, statistics.getDroppedFrames());

            m_droppedFrameAverage.reset();
            m_dropped.insert(quality);
        }
    }

    m_lastUpdate = now;
    m_lastDroppedFrames = statistics.getDroppedFrames();
    m_lastDecodedFrames = statistics.getDecodedFrames();
}

void DroppedFrameFilter::onStreamChange()
{
    m_lastDecodedFrames = 0;
    m_lastDroppedFrames = 0;
    m_droppedFrameAverage.reset();
    m_lastUpdate = MediaTime::zero();
    m_dropped.clear();
}
}
}
