#include "Percentile.hpp"

namespace twitch {
WeightedPercentile::WeightedPercentile(double maxWeight)
    : m_maxWeight(maxWeight)
{
    reset();
}

void WeightedPercentile::add(double weight, double value)
{
    sort(SortOrder::ByIndex);
    Sample sample;
    sample.index = m_index++;
    sample.weight = weight;
    sample.value = value;
    m_samples.push_back(sample);
    m_currentWeight += weight;

    while (m_currentWeight > m_maxWeight && !m_samples.empty()) {
        double excess = m_currentWeight - m_maxWeight;
        auto& first = m_samples.front();
        if (first.weight <= excess) {
            m_currentWeight -= first.weight;
            m_samples.erase(m_samples.begin());
        } else {
            first.weight -= excess;
            m_currentWeight = m_currentWeight - excess;
        }
    }
}

double WeightedPercentile::percentile(double percentile)
{
    sort(SortOrder::ByValue);
    double total = 0;
    for (const auto& sample : m_samples) {
        total += sample.weight;
        if (total >= percentile * m_currentWeight) {
            return sample.value;
        }
    }
    return m_samples.empty() ? std::numeric_limits<double>::quiet_NaN() : m_samples.back().value;
}

void WeightedPercentile::reset()
{
    m_currentWeight = 0;
    m_index = 0;
    m_order = SortOrder::Unsorted;
    m_samples.clear();
}

void WeightedPercentile::sort(SortOrder order)
{
    if (m_order != order) {
        std::sort(m_samples.begin(), m_samples.end(),
            [order](const Sample& a, const Sample& b) {
                return order == SortOrder::ByIndex ? a.index < b.index : a.value < b.value;
            });
        m_order = order;
    }
}
}
