#pragma once

#include "CircularQueue.hpp"
#include "Estimator.hpp"
#include "Filter.hpp"
#include "RequestMetric.hpp"
#include "debug/PrefixLog.hpp"
#include "player/BufferState.hpp"
#include "player/Qualities.hpp"
#include "player/Settings.hpp"
#include "playercore/platform/Platform.hpp"

namespace twitch {
namespace abr {
/** Adaptive quality selector */
class QualitySelector : public MediaSource::Request::Listener, private Filter::Context {
public:
    explicit QualitySelector(std::shared_ptr<Platform> platform);
    ~QualitySelector() override = default;
    void onRequestSent(const MediaSource::Request& request) override;
    void onResponseReceived(const MediaSource::Request& request) override;
    void onResponseBytes(const MediaSource::Request& request, size_t bytes) override;
    void onResponseEnd(const MediaSource::Request& request) override;
    void onRequestError(const MediaSource::Request& request, int error) override;
    void onBufferDurationChange(const TimeRange& range);
    void onBufferStateChange(BufferState state);

    void onStatistics(const Statistics& statistics, const Quality& quality);
    void onStreamChange();

    const Quality& nextQuality(const Qualities& qualities);
    bool canReplaceBuffer(const Qualities& qualities, MediaTime duration);

    int getAverageBitrate() const;
    int getBandwidthEstimate() const;
    void setInitialBitrate(int bitrate);
    void setLowLatencyMode(bool enable);
    void setMaxBitrate(int bitrate);
    void setMaxVideoSize(int width, int height);
    void setViewportSize(int width, int height);
    void setViewportScale(float scale);
    void setPlaybackRate(float rate);

    enum class StreamType {
        Live,
        VOD,
    };
    void setStreamType(StreamType type);
    void setTargetBufferSize(MediaTime buffered);

    // for reporting only
    const CircularQueue<RequestMetric>& getTransferHistory() const;
    // filter context
    BufferState getBufferState() const override { return m_state; }
    MediaTime getBufferDuration() const override { return m_buffered; }
    MediaTime getMinBufferTarget() const override;
    float getPlaybackRate() const override { return m_playbackRate; }
    const Quality& getSelected() const override { return m_selected; }
    Quality getTarget(const std::vector<Quality>& qualities) const override;
    bool isLive() const override { return m_type == StreamType::Live; }
    void filter(const Filter& filter, const Quality& quality) override;

private:
    template <typename Filter, typename Method, typename... Args>
    void filter(Method method, Args&&... args)
    {
        for (auto& f : m_filters) {
            if (f->getName() == Filter::Name) {
                Filter* ptr = static_cast<Filter*>(f.get());
                if (ptr) {
                    (*ptr.*method)(std::forward<Args>(args)...);
                }
            }
        }
    }

    PrefixedLog m_log;
    std::vector<std::unique_ptr<Filter>> m_filters;
    std::set<std::string> m_disabledFilters;
    MediaTime m_buffered;
    float m_playbackRate;
    Quality m_selected;
    BufferState m_state;
    StreamType m_type;
    std::set<Quality> m_filtered;
    std::string m_filterlog;
};
}
}
