#include "ViewportFilter.hpp"
#include <algorithm>
#include <limits>

namespace twitch {
namespace abr {
std::string ViewportFilter::Name = "ViewportFilter";
const int MaxPixelArea = std::numeric_limits<int>::max();
const int MinPixelArea = 409920; // 854 * 480;

ViewportFilter::ViewportFilter()
    : m_pixelArea(MaxPixelArea)
    , m_scale(1.5)
{
}

bool ViewportFilter::filter(const std::vector<Quality>& qualities, Filter::Context& context)
{
    auto comp = [](int area, const Quality& quality) {
        return quality.width * quality.height > area;
    };
    auto upper = std::upper_bound(qualities.rbegin(), qualities.rend(), m_pixelArea, comp);

    if (upper != qualities.rend()) {
        int maxArea = upper->width * upper->height;
        for (auto it = upper; it < qualities.rend(); it++) {
            int area = it->width * it->height;
            if (area > maxArea) {
                context.filter(*this, *it);
            }
        }
    }
    return true;
}

void ViewportFilter::onStatistics(const Statistics& statistics, const Quality& quality)
{
    (void)statistics;
    (void)quality;
}

void ViewportFilter::setViewportSize(int width, int height)
{
    if (width < MaxPixelArea) {
        // hack to never go below 720p
        int pixelArea = std::max(MinPixelArea, width * height);
        m_pixelArea = static_cast<int>(pixelArea * m_scale);
    }
}

void ViewportFilter::setViewportScale(float scale)
{
    m_scale = scale > 0.0f ? scale : 1.0f;
}
}
}
