#include "AnalyticsTracker.hpp"
#include "AutoQualityChanged.hpp"
#include "BufferEmpty.hpp"
#include "BufferNSeconds.hpp"
#include "BufferRefill.hpp"
#include "MinuteWatched.hpp"
#include "NSecondsWatched.hpp"
#include "VideoEnd.hpp"
#include "VideoError.hpp"
#include "VideoPlay.hpp"
#include "VideoSeekSuccess.hpp"
#include "util/Random.hpp"
#include "util/Uuid.hpp"

namespace twitch {
namespace analytics {
AnalyticsTracker::AnalyticsTracker(Player& player,
    Player::Listener& listener,
    std::shared_ptr<Platform> platform,
    std::shared_ptr<Scheduler> scheduler)
    : m_player(player)
    , m_listener(listener)
    , m_platform(platform)
    , m_spadeClient(platform->getHttpClient(), SpadeClient::DefaultUrl)
    , m_sendEvents(false)
    , m_enabled(true)
    , m_keepPlaySession(false)
    , m_bufferEmptyCount(0)
{
    m_events.emplace_back(new AutoQualityChanged(m_player, *this));
    m_events.emplace_back(new BufferEmpty(*this));
    m_events.emplace_back(new BufferRefill(*this));
    m_events.emplace_back(new BufferNSeconds(*this, scheduler, MediaTime(2.0)));
    m_events.emplace_back(new MinuteWatched(*this, m_player, scheduler));
    std::vector<MediaTime> watchTimes { MediaTime(5), MediaTime(15), MediaTime(45) };
    m_events.emplace_back(new NSecondsWatched(*this, scheduler, watchTimes));
    m_events.emplace_back(new VideoEnd(*this));
    m_events.emplace_back(new VideoError(*this));
    m_events.emplace_back(new VideoPlay(*this));
    m_events.emplace_back(new VideoSeekSuccess(*this));
}

void AnalyticsTracker::onDurationChanged(MediaTime duration)
{
    if (m_session && !m_session->hasContentType()) {
        bool isLive = duration == MediaTime::max();
        PlaySession::ContentType type = isLive
            ? PlaySession::ContentType::Live
            : PlaySession::ContentType::VOD;
        m_session->setContentType(type);
    }
    processEvent(&AnalyticsEvent::onDurationChanged, MediaTime::now<Clock>(), duration);
}

void AnalyticsTracker::onError(const Error& error)
{
    processEvent(&AnalyticsEvent::onError, error);
}

void AnalyticsTracker::onPlayerLoad(const std::string& path)
{
    auto now = MediaTime::now<AnalyticsTracker::Clock>();

    m_path = path;
    m_bufferEmptyCount = 0;

    if (!m_session || !m_keepPlaySession) {
        onRegeneratePlaySession(now);
    }

    processEvent(&AnalyticsEvent::onPlayerLoad, now, path, m_keepPlaySession);
}

void AnalyticsTracker::onPlayerSeek(MediaTime origin, MediaTime dest)
{
    processEvent(&AnalyticsEvent::onPlayerSeek, origin, dest);
}

void AnalyticsTracker::onRegeneratePlaySession()
{
    auto now = MediaTime::now<AnalyticsTracker::Clock>();
    onRegeneratePlaySession(now);
}

void AnalyticsTracker::onRegeneratePlaySession(MediaTime timestamp)
{
    // Create the Session before processing the events since the Session could
    // be used to populate the event properties
    m_session.reset(new PlaySession(m_path));

    processEvent(&AnalyticsEvent::onPlaySession, *m_session, timestamp);
}

void AnalyticsTracker::onEventReady(const AnalyticsEvent& event, json11::Json::object& properties)
{
    if (m_enabled) {
        populateProperties(properties);
        m_listener.onAnalyticsEvent(event.getName(), json11::Json(properties).dump());

        if (m_sendEvents) {
            m_spadeClient.send(event.getName(), properties);
        }
    }
}

void AnalyticsTracker::onMetadata(const std::string& type, const std::vector<uint8_t>& data)
{
    (void)type;
    (void)data;
}

void AnalyticsTracker::onQualityChanged(const Quality& quality)
{
    processEvent(&AnalyticsEvent::onQualityChanged, quality);
}

void AnalyticsTracker::onPositionChanged(MediaTime)
{
}

void AnalyticsTracker::onRebuffering()
{
    auto timestamp = MediaTime::now<Clock>();
    m_bufferEmptyCount++;
    std::string bufferSessionId = Uuid::random().toString(AnalyticsEvent::uuidFormat);
    processEvent(&AnalyticsEvent::onRebuffering, timestamp, bufferSessionId, m_bufferEmptyCount);
}

void AnalyticsTracker::onRecoverableError(const Error& error)
{
    processEvent(&AnalyticsEvent::onRecoverableError, error);
}

void AnalyticsTracker::onSessionData(const std::map<std::string, std::string>& properties)
{
    m_sessionData = properties;
}

void AnalyticsTracker::onStateChanged(Player::State state)
{
    auto timestamp = MediaTime::now<Clock>();
    processEvent(&AnalyticsEvent::onStateChanged, timestamp, state);
}

void AnalyticsTracker::populateProperties(json11::Json::object& properties)
{
    // Platform
    properties["platform"] = m_platform->getName();
    properties["player"] = m_platform->getName();
    properties["device_id"] = m_deviceId;
    properties["backend"] = m_player.getName();
    properties["core_version"] = m_player.getVersion();

    // Player settings
    properties["muted"] = m_player.isMuted();
    properties["volume"] = m_player.getVolume();

    const auto& path = m_player.getPath();
    if (!path.empty()) {
        properties["url"] = path;
    }

    if (m_session) {
        m_session->addProperties(properties);
    }

    auto position = m_player.getPosition();
    auto bufferPosition = m_player.getBufferedPosition();
    if (position <= bufferPosition) {
        auto bufferSize = bufferPosition - position;
        properties["video_buffer_size"] = bufferSize.seconds();
    }

    const auto quality = m_player.getQuality();

    if (m_player.getAutoSwitchQuality()) {
        properties["quality"] = "auto";
    } else {
        if (!quality.name.empty()) {
            properties["quality"] = quality.name;
        }
    }

    if (!quality.group.empty()) {
        properties["stream_format"] = quality.group;
        properties["current_bitrate"] = quality.bitrate;
    }

    if (quality.width > 0 && quality.height > 0) {
        properties["vid_height"] = quality.height;
        properties["vid_width"] = quality.width;
    }

    const auto& stats = m_player.getStatistics();
    if (stats.getFrameRate() > 0) {
        properties["current_fps"] = stats.getFrameRate();
    }

    const auto& sessionData = m_sessionData;
    auto setPlaylistProperty = [&properties, &sessionData](const std::string& eventProperty, const std::string& playlistProperty) {
        auto itr = sessionData.find(playlistProperty);
        if (itr != sessionData.end() && !itr->second.empty()) {
            // handle true/false strings as bool
            if (itr->second == "true") {
                properties[eventProperty] = true;
            } else if (itr->second == "false") {
                properties[eventProperty] = false;
            } else {
                properties[eventProperty] = itr->second;
            }
        }
    };

    setPlaylistProperty("cluster", "CLUSTER");
    setPlaylistProperty("node", "NODE");
    setPlaylistProperty("manifest_cluster", "MANIFEST-CLUSTER");
    setPlaylistProperty("manifest_node", "MANIFEST-NODE");
    setPlaylistProperty("manifest_node_type", "MANIFEST-NODE-TYPE");
    setPlaylistProperty("serving_id", "SERVING-ID");
    setPlaylistProperty("cluster", "CLUSTER");
    setPlaylistProperty("transcoder_type", "TRANSCODESTACK");
    setPlaylistProperty("broadcast_id", "BROADCAST-ID");
    setPlaylistProperty("low_latency", "FUTURE");

    if (m_session && !m_session->isLive()) {
        properties["vod_timestamp"] = position.seconds();
        setPlaylistProperty("vod_cdn_origin", "ORIGIN");
        setPlaylistProperty("vod_cdn_region", "REGION");
    }

    // Overwrite any previously set properties with
    // more important platform specific properties.
    const std::map<std::string, std::string>& analyticsProperties = m_platform->getAnalyticsProperties();
    for (const auto& props : analyticsProperties) {
        properties[props.first] = props.second;
    }
}
}
}
