#pragma once

#include "AnalyticsEvent.hpp"
#include "PlaySession.hpp"
#include "SpadeClient.hpp"
#include "playercore/Player.hpp"
#include "playercore/platform/Scheduler.hpp"
#include <json11.hpp>
#include <memory>
#include <vector>

namespace twitch {
class MediaPlayer;
namespace analytics {
/** AnalyticsTracker helps the MediaPlayer emit its Player::onAnalyticsEvent events */
class AnalyticsTracker : public Player::Listener, private analytics::AnalyticsEvent::Listener {
public:
    AnalyticsTracker(Player& player,
        Player::Listener& listener,
        std::shared_ptr<Platform> platform,
        std::shared_ptr<Scheduler> scheduler);
    ~AnalyticsTracker() override = default;
    void setSendEvents(bool enabled) { m_sendEvents = enabled; }
    void setAnalyticsEndpoint(const std::string& endpoint) { m_spadeClient.setAnalyticsEndpoint(endpoint); }
    void setEnabled(bool enabled) { m_enabled = enabled; }

    // Player::Listener overrides
    void onDurationChanged(MediaTime duration) override;
    void onError(const Error& error) override;
    void onMetadata(const std::string& type, const std::vector<uint8_t>& data) override;
    void onQualityChanged(const Quality& quality) override;
    void onRebuffering() override;
    void onRecoverableError(const Error& error) override;
    void onSeekCompleted(MediaTime) override {}
    void onSessionData(const std::map<std::string, std::string>& properties) override;
    void onStateChanged(Player::State state) override;
    void onAnalyticsEvent(const std::string&, const std::string&) override {}
    void onPositionChanged(MediaTime position) override;

    using Clock = std::chrono::system_clock;
    // Player API
    void onPlayerLoad(const std::string& path);
    void onPlayerSeek(MediaTime origin, MediaTime dest);

    // PlaySession
    void onRegeneratePlaySession();
    void onRegeneratePlaySession(MediaTime timestamp);
    void setKeepPlaySession(bool keepPlaySession) { m_keepPlaySession = keepPlaySession; }
    void setDeviceId(const std::string& id) { m_deviceId = id; }

private:
    void onEventReady(const analytics::AnalyticsEvent& event, json11::Json::object& properties) override;
    void populateProperties(json11::Json::object& properties);

    template <typename Method, typename... Args>
    void processEvent(Method method, Args&&... args)
    {
        for (auto& event : m_events) {
            (*event.*method)(std::forward<Args>(args)...);
        }
    }

    Player& m_player;
    Player::Listener& m_listener;
    std::shared_ptr<Platform> m_platform;
    std::unique_ptr<PlaySession> m_session;
    std::vector<std::unique_ptr<AnalyticsEvent>> m_events;
    SpadeClient m_spadeClient;
    bool m_sendEvents;
    bool m_enabled;
    std::string m_path;
    bool m_keepPlaySession;
    std::string m_deviceId;
    int m_bufferEmptyCount;
    std::map<std::string, std::string> m_sessionData;
};
}
}
